/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.api;

import org.jboss.identity.idm.exception.IdentityException;

import java.util.Collection;

/**
 * Management of relationships between Identity and Group objects.
 *
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public interface RelationshipManager
{

   /**
    * @return Session associated with this object instance
    */
   IdentitySession getIdentitySession();

   /**
    * @return
    */
   RelationshipManagerFeaturesDescription getFeaturesDescription();

   // Assignation

    /**
    * <p>Associate groups</p>
    * @param parents
    * @param members
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   void associateGroups(Collection<Group> parents, Collection<Group> members)
   throws IdentityException;

   /**
    * <p>Associate groups</p>
    * @param parent
    * @param member
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   void associateGroups(Group parent, Group member)
   throws IdentityException;

   /**
    * <p>Associate identities to groups</p>
    * @param parents
    * @param members
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   void associateIdentities(Collection<Group> parents, Collection<Identity> members)
   throws IdentityException;

   /**
    * <p>Associate identities to groups</p>
    * @param parents
    * @param members
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   void associateIdentities(Group parents, Identity members)
   throws IdentityException;

   /**
    * <p>Disassociate groups</p>
    * @param parents
    * @param members
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   void disassociateGroups(Collection<Group> parents, Collection<Group> members)
   throws IdentityException;

   /**
    * <p>Disassociate identities from groups</p>
    * @param parents
    * @param members
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   void disassociateIdentities(Collection<Group> parents, Collection<Identity> members)
   throws IdentityException;

   /**
    * <p>Check if association is present </p>
    * @param parents
    * @param members
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   <G extends IdentityType, I extends IdentityType> boolean isAssociated(Collection<G> parents, Collection<I> members)
   throws IdentityException;

   /**
    * <p>Check if association is present </p>
    * @param parent
    * @param member
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   <G extends IdentityType, I extends IdentityType> boolean isAssociated(G parent, I member)
   throws IdentityException;

   // Resolve relationships

   /**
    * Find groups that are associated with given group.
    * If the paginatedSearch or orderedSearch operations
    * are not supported, dedicated parameters will
    * take no effect. If 'parent' parameter is set to false, all parent group will be returned. If parent parameter is
    * set to true and 'inherited' is set to true all nested subgroubs will be returned.
    *
    * @param group parent group
    * @param groupType can be null
    * @param parent defines if given identity is parent or child side in the
    * relationship - default is true (parent)
    * @param inherited if true also identities from subgroubs will be retreived. Matters only when parent is set to true.
    * Default is false
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Collection<Group> findAssociatedGroups(Group group,
                                        GroupType groupType,
                                        boolean parent,
                                        boolean inherited) throws IdentityException;

   /**
    * Find groups that are associated with given group.
    * If the paginatedSearch or orderedSearch operations
    * are not supported, dedicated parameters will
    * take no effect. If 'parent' parameter is set to false, all parent group will be returned. If parent parameter is
    * set to true and 'inherited' is set to true all nested subgroubs will be returned.
    *
    * @param group parent group
    * @param groupType can be null
    * @param parent defines if given identity is parent or child side in the
    * relationship - default is true (parent)
    * @param inherited if true also identities from subgroubs will be retreived. Matters only when parent is set to true.
    * Default is false
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Collection<Group> findAssociatedGroups(Group group,
                                        GroupType groupType,
                                        boolean parent,
                                        boolean inherited,
                                        IdentitySearchControl[] controls) throws IdentityException;

    /**
    * Find all groups that given identity is associated with.
    * If the paginatedSearch or orderedSearch operations
    * are not supported, dedicated parameters will
    * take no effect
    *
    * @param identity child identity
    * @param groupType can be null
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Collection<Group> findAssociatedGroups(Identity identity,
                                          GroupType groupType) throws IdentityException;

   /**
    * Find all groups that given identity is associated with.
    * If the paginatedSearch or orderedSearch operations
    * are not supported, dedicated parameters will
    * take no effect
    *
    * @param identity child identity
    * @param groupType can be null
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Collection<Group> findAssociatedGroups(Identity identity,
                                          GroupType groupType,
                                          IdentitySearchControl[] controls) throws IdentityException;

   /**
    * Find all groups that given identity is associated with.
    * If the paginatedSearch or orderedSearch operations
    * are not supported, dedicated parameters will
    * take no effect
    *
    * @param identity child identity
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Collection<Group> findAssociatedGroups(Identity identity,
                                          IdentitySearchControl[] controls) throws IdentityException;

   /**
    * Find all groups that given identity is associated with.
    * If the paginatedSearch or orderedSearch operations
    * are not supported, dedicated parameters will
    * take no effect
    *
    * @param identity child identity
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Collection<Group> findAssociatedGroups(Identity identity) throws IdentityException;

   /**
    * Find identities that have relationship with given parent group.
    * If the paginatedSearch or orderedSearch operations
    * are not supported in this store implementation, dedicated parameters will
    * take no effect
    *
    * @param group parent group
    * @param inherited if true also identities from subgroubs will be retrieved. Default is false
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Collection<Identity> findAssociatedIdentities(Group group,
                                           boolean inherited) throws IdentityException;

   /**
    * Find identities that have relationship with given parent group.
    * If the paginatedSearch or orderedSearch operations
    * are not supported in this store implementation, dedicated parameters will
    * take no effect
    *
    * @param group parent group
    * @param inherited if true also identities from subgroubs will be retrieved. Default is false
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Collection<Identity> findAssociatedIdentities(Group group,
                                           boolean inherited,
                                           IdentitySearchControl[] controls) throws IdentityException;

}

