/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.api;

import org.jboss.identity.idm.exception.IdentityException;

import java.util.Collection;

/**
 * Management operations on Role objects. Role support is optional and depends on the capabilities of the underlying
 * configured identity stores capabilities.
 *
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public interface RoleManager
{

   /**
    * @return Session associated with this object instance
    */
   IdentitySession getIdentitySession();

   /**
    * @return
    */
   RoleManagerFeaturesDescription getSupportedFeatures();

   // RoleType

   /**
    * Create RoleType
    * @param name
    * @return
    * @throws IdentityException
    */
   RoleType createRoleType(String name) throws IdentityException;

   /**
    * Remove RoleType
    * @param name
    * @throws IdentityException
    */
   void removeRoleType(String name) throws IdentityException;

   /**
    * Remove RoleType
    * @param roleType
    * @throws IdentityException
    */
   void removeRoleType(RoleType roleType) throws IdentityException;

   /**
    * Get RoleType
    * @param name
    * @return
    * @throws IdentityException
    */
   RoleType getRoleType(String name) throws IdentityException;

   /**
    * Find all RoleType objects stored
    * @return
    * @throws IdentityException
    */
   Collection<RoleType> findRoleTypes(IdentitySearchControl[] controls) throws IdentityException;


   // Role
   /**
    * Create role
    * @param roleType
    * @param identity
    * @param group
    * @return
    * @throws IdentityException
    */
   Role createRole(RoleType roleType, Identity identity, Group group) throws IdentityException;

   /**
    * Remove Role
    * @param roleType
    * @param identity
    * @param group
    * @throws IdentityException
    */
   void removeRole(RoleType roleType, Identity identity, Group group) throws IdentityException;

   /**
    * Remove Role
    * @param role
    * @throws IdentityException
    */
   void removeRole(Role role) throws IdentityException;

   /**
    * Check if Role is present
    * @param identity
    * @param group
    * @param roleType
    * @return
    * @throws IdentityException
    */
   boolean hasRole(Identity identity, Group group, RoleType roleType) throws IdentityException;

   /**
    * Find RoleType objects for roles associated with a given Identity and Group
    * @param identity
    * @param group
    * @return
    * @throws IdentityException
    */
   Collection<RoleType> findRoleTypes(Identity identity, Group group,
                                      IdentitySearchControl[] controls) throws IdentityException;

   /**
    * Find RoleType objects for roles associated with a given Identity and Group
    * @param identity
    * @param group
    * @return
    * @throws IdentityException
    */
   Collection<RoleType> findRoleTypes(Identity identity, Group group) throws IdentityException;

   /**
    * Find RoleType objects for roles associated with a given Identity
    * @param identity
    * @return
    * @throws IdentityException
    */
   Collection<RoleType> findIdentityRoleTypes(Identity identity) throws IdentityException;

    /**
    * Find RoleType objects for roles associated with a given Identity
    * @param identity
    * @return
    * @throws IdentityException
    */
   Collection<RoleType> findIdentityRoleTypes(Identity identity, IdentitySearchControl[] controls) throws IdentityException;

   /**
    * Find RoleType objects for roles associated with a given Group
    * @param group
    * @return
    * @throws IdentityException
    */
   Collection<RoleType> findGroupRoleTypes(Group group) throws IdentityException;

   /**
    * Find RoleType objects for roles associated with a given Group
    * @param group
    * @return
    * @throws IdentityException
    */
   Collection<RoleType> findGroupRoleTypes(Group group,
                                      IdentitySearchControl[] controls) throws IdentityException;

   /**
    * Find Role objects with a given RoleType associated with a given IdentityType
    * @param identityType
    * @param roleType
    * @return
    * @throws IdentityException
    */
   <T extends IdentityType> Collection<Role> findRoles(T identityType,
                                                       RoleType roleType) throws IdentityException;

}
