/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.api;

import org.jboss.identity.idm.exception.IdentityException;

import java.util.Collection;

/**
 * Exposes all management operations on Group and Identity objects.
 *
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public interface PersistenceManager
{

   /**
    * @return Session associated with this object instance
    */
   IdentitySession getIdentitySession();


   /**
    * @return object describing supported features
    */
   PersistenceManagerFeaturesDescription getFeaturesDescription();


   // Create

   /**
    * <p>Create an identity in the realm</p>
    * @param identityName
    * @throws IdentityException
    * @return
    */
   Identity createIdentity(String identityName) throws IdentityException;

   /**
    * <p>Create a group of a particular type</p>
    * @param groupName
    * @param groupType
    * @return
    */
   Group createGroup(String groupName, GroupType groupType)
   throws IdentityException;

   // Remove

   /**
    * Remove given identity
    *
    * @param identity
    * @param force - if true all present relationships will be removed, if false any present relationship will cause
    * removal to fail 
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   void removeIdentity(Identity identity, boolean force) throws IdentityException;

   /**
    * Remove given group
    *
    * @param group
    * @param force - if true all present relationships will be removed, if false any present relationship will cause
    * removal to fail
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   void removeGroup(Group group, boolean force) throws IdentityException;

   // Search

   /**
    * @return a number of stored identities
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   int getIdentityCount()
      throws IdentityException;

   /**
    * @param groupType
    * @return a number of stored groups with a given type
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   int getGroupTypeCount(GroupType groupType)
      throws IdentityException;

   /**
    * Find identity with a given name
    *
    * @param name
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Identity findIdentity(String name) throws IdentityException;

   /**
    * Find identity with a given id
    *
    * @param id
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Identity findIdentityById(String id) throws IdentityException;

   /**
    *
    * @param controls
    * @return
    * @throws IdentityException
    */
   Collection<Identity> findIdentity(IdentitySearchControl[] controls) throws IdentityException;

   // Search Groups

   /**
    * Find group with a given name and type
    *
    * @param name
    * @param groupType
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Group findGroup(String name, GroupType groupType) throws IdentityException;

   /**
    * Find group with a given id
    *
    * @param id
    * @return
    * @throws org.jboss.identity.idm.exception.IdentityException
    */
   Group findGroupById(String id) throws IdentityException;


   /**
    *
    * @param groupType
    * @param controls
    * @return
    * @throws IdentityException
    */
   Collection<Group> findGroup(GroupType groupType, IdentitySearchControl[] controls) throws IdentityException;

   /**
    * 
    * @param groupType
    * @return
    * @throws IdentityException
    */
   Collection<Group> findGroup(GroupType groupType) throws IdentityException;


   /**
    * Whether Identity is a virtual
    * (such as a pseudonym)
    * @return true (virtual identity)
    */
   boolean isVirtual(Identity identity);



   

}
