/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api;

import org.jboss.identity.idm.api.IdentitySessionFactory;
import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityConfigurationMetaData;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityStoreConfigurationMetaData;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityRepositoryConfigurationMetaData;
import org.jboss.identity.idm.spi.configuration.metadata.RealmConfigurationMetaData;
import org.jboss.identity.idm.spi.store.IdentityStore;
import org.jboss.identity.idm.spi.store.AttributeStore;
import org.jboss.identity.idm.spi.repository.IdentityStoreRepository;
import org.jboss.identity.idm.impl.configuration.jaxb2.JAXB2IdentityConfiguration;
import org.jboss.identity.idm.impl.api.session.IdentitySessionImpl;
import org.jboss.identity.idm.impl.api.session.mapper.DirectIdentityObjectTypeMapperImpl;
import org.jboss.identity.idm.impl.api.session.mapper.IdentityObjectTypeMapper;
import org.jboss.identity.idm.impl.api.session.mapper.IdentityObjectTypeMapperImpl;
import org.jboss.identity.idm.exception.IdentityException;

import java.util.Map;
import java.util.HashMap;
import java.io.File;
import java.lang.reflect.Constructor;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class IdentitySessionFactoryImpl implements IdentitySessionFactory
{

   private final Map<String, IdentitySession> realmMap;

   public IdentitySessionFactoryImpl(Map<String, IdentitySession> realmMap)
   {
      this.realmMap = realmMap;
   }

   public IdentitySessionFactoryImpl(IdentityConfigurationMetaData configMD) throws Exception
   {
      realmMap = createSessionMap(configMD); 
   }

   public IdentitySessionFactoryImpl(File configFile) throws Exception
   {
      IdentityConfigurationMetaData configMD = JAXB2IdentityConfiguration.createConfigurationMetaData(configFile);

      realmMap = createSessionMap(configMD);

   }

   private Map<String, IdentitySession> createSessionMap(IdentityConfigurationMetaData configMD) throws Exception
   {
      //TODO: some validation, sanity checks and error reporting

      // IdentityStore

      Map<String, IdentityStore> bootstrappedIdentityStores = new HashMap<String, IdentityStore>();
      Map<String, AttributeStore> bootstrappedAttributeStores = new HashMap<String, AttributeStore>();

      for (IdentityStoreConfigurationMetaData metaData : configMD.getIdentityStores())
      {
         Class storeClass = null;
         try
         {
            storeClass = Class.forName(metaData.getClassName());
         }
         catch (ClassNotFoundException e)
         {
            throw new IdentityException("Cannot instantiate identity store:" + metaData.getClassName(), e);
         }
         Class partypes[] = new Class[1];
         partypes[0] = String.class;

         Constructor ct = storeClass.getConstructor(partypes);
         Object argList[] = new Object[1];
         argList[0] = metaData.getId();

         IdentityStore store = (IdentityStore)ct.newInstance(argList);

         store.bootstrap(metaData);

         bootstrappedIdentityStores.put(store.getId(), store);
         bootstrappedAttributeStores.put(store.getId(), store);
      }

      // IdentityRepository

      Map<String, IdentityStoreRepository> bootstrappedRepositories = new HashMap<String, IdentityStoreRepository>();

      for (IdentityRepositoryConfigurationMetaData metaData : configMD.getRepositories())
      {
         Class repoClass = null;
         try
         {
            repoClass = Class.forName(metaData.getClassName());
         }
         catch (ClassNotFoundException e)
         {
            throw new IdentityException("Cannot instantiate identity store:" + metaData.getClassName(), e);
         }
         Class partypes[] = new Class[1];
         partypes[0] = String.class;

         Constructor ct = repoClass.getConstructor(partypes);
         Object argList[] = new Object[1];
         argList[0] = metaData.getId();

         IdentityStoreRepository repo = (IdentityStoreRepository)ct.newInstance(argList);

         repo.bootstrap(metaData, bootstrappedIdentityStores, bootstrappedAttributeStores);

         bootstrappedRepositories.put(repo.getId(), repo);
      }

      // Realms

      Map<String, IdentitySession> sessionMap = new HashMap<String, IdentitySession>();

      for (RealmConfigurationMetaData metaData : configMD.getRealms())
      {
         String realmName = metaData.getId();

         IdentityStoreRepository repo = bootstrappedRepositories.get(metaData.getIdentityRepositoryIdRef());

         IdentityObjectTypeMapper mapper = null;

         if (metaData.getGroupTypeMappings() == null || metaData.getGroupTypeMappings().isEmpty())
         {
            // use direct type mapper
            mapper = new DirectIdentityObjectTypeMapperImpl(metaData.getIdentityMapping());
         }
         else
         {
            mapper = new IdentityObjectTypeMapperImpl(metaData.getGroupTypeMappings(), metaData.getIdentityMapping());
         }

         IdentitySession session = new IdentitySessionImpl(realmName, repo, mapper);

         sessionMap.put(realmName, session);
      }

      return sessionMap;
   }

   public void close()
   {

   }

   public boolean isClosed()
   {
      return false;
   }

   public IdentitySession createIdentitySession(String realmName)
   {
      return realmMap.get(realmName);
   }

   public IdentitySession getCurrentIdentitySession(String realmName)
   {
      return realmMap.get(realmName);
   }
}
