/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session.managers;

import org.jboss.identity.idm.api.PersistenceManager;
import org.jboss.identity.idm.api.Identity;
import org.jboss.identity.idm.api.Group;
import org.jboss.identity.idm.api.GroupType;
import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.PersistenceManagerFeaturesDescription;
import org.jboss.identity.idm.api.IdentitySearchControl;
import org.jboss.identity.idm.exception.IdentityException;
import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.searchcontrol.IdentityObjectSearchControl;
import org.jboss.identity.idm.impl.NotYetImplementedException;
import org.jboss.identity.idm.impl.api.session.managers.AbstractManager;

import java.util.Collection;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class PersistenceManagerImpl extends AbstractManager implements PersistenceManager
{

   private final PersistenceManagerFeaturesDescription featuresDescription;


   public PersistenceManagerImpl(IdentitySession session)
   {
      super(session);

      featuresDescription = new PersistenceManagerFeaturesDescription()
      {
         public boolean isIdentitiesAddRemoveSupported()
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
               isIdentityObjectAddRemoveSupported(objectType);
         }

         public boolean isGroupsAddRemoveSupported(GroupType groupType)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);

            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
               isIdentityObjectAddRemoveSupported(objectType);
         }

         public boolean isIdentitiesSearchControlSupported(IdentitySearchControl control)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

            if (control instanceof IdentityObjectSearchControl)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isControlSupported(objectType, (IdentityObjectSearchControl)control);
            }

            return false;
         }


         public boolean isIdentitiesSearchControlSupported(Class controlClazz)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
               isControlSupported(objectType, controlClazz);

         }

         public boolean isGroupsSearchControlSupported(GroupType groupType, IdentitySearchControl control)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);

            if (control instanceof IdentityObjectSearchControl)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isControlSupported(objectType, (IdentityObjectSearchControl)control);
            }

            return false;
         }

         public boolean isGroupsSearchControlSupported(GroupType groupType, Class controlClazz)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);

            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
               isControlSupported(objectType, controlClazz);
         }
      };
   }

   public PersistenceManagerFeaturesDescription getFeaturesDescription()
   {
      return featuresDescription;
   }

   public Identity createIdentity(String identityName) throws IdentityException
   {
      IdentityObjectType iot = getIdentityObjectType();

      IdentityObject identityObject = getRepository().createIdentityObject(getInvocationContext(), identityName, iot);

      return createIdentity(identityObject);
   }

   public Group createGroup(String groupName, GroupType groupType) throws IdentityException
   {
      IdentityObjectType iot = getIdentityObjectType(groupType);

      IdentityObject identityObject = getRepository().createIdentityObject(getInvocationContext(), groupName, iot);

      return createGroup(identityObject);
   }

   public void removeIdentity(Identity identity, boolean force) throws IdentityException
   {
      IdentityObjectType iot = getIdentityObjectType();

      getRepository().removeIdentityObject(getInvocationContext(), createIdentityObject(identity));
      
   }

   public void removeGroup(Group group, boolean force) throws IdentityException
   {
      //TODO: force

      getRepository().removeIdentityObject(getInvocationContext(), createIdentityObject(group));
   }

   public int getIdentityCount() throws IdentityException
   {
      IdentityObjectType iot = getIdentityObjectType();

      return getRepository().getIdentityObjectsCount(getInvocationContext(), iot);
   }

   public int getGroupTypeCount(GroupType groupType) throws IdentityException
   {
      IdentityObjectType iot = getIdentityObjectType(groupType);

      return getRepository().getIdentityObjectsCount(getInvocationContext(), iot);
   }

   public Identity findIdentity(String name) throws IdentityException
   {
      return createIdentity(getRepository().findIdentityObject(getInvocationContext(), name, getIdentityObjectType()));
   }

   public Identity findIdentityById(String id) throws IdentityException
   {
      return createIdentity(getRepository().findIdentityObject(getInvocationContext(), id));
   }

   public Collection<Identity> findIdentity(IdentitySearchControl[] controls) throws IdentityException
   {


      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), getIdentityObjectType(), convertSearchControls(controls));
      List<Identity> identities = new LinkedList<Identity>();

      for (Iterator<IdentityObject> iterator = ios.iterator(); iterator.hasNext();)
      {
         IdentityObject identityObject = iterator.next();
         identities.add(createIdentity(identityObject));
      }

      return identities;
   }

   public Group findGroup(String name, GroupType groupType) throws IdentityException
   {
      return createGroup(getRepository().findIdentityObject(getInvocationContext(), name, getIdentityObjectType(groupType)));
   }

   public Group findGroupById(String id) throws IdentityException
   {
      return createGroup(getRepository().findIdentityObject(getInvocationContext(), id));
   }

   public Collection<Group> findGroup(GroupType groupType, IdentitySearchControl[] controls) throws IdentityException
   {
      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), getIdentityObjectType(groupType), convertSearchControls(controls));
      List<Group> groups = new LinkedList<Group>();

      for (Iterator<IdentityObject> iterator = ios.iterator(); iterator.hasNext();)
      {
         IdentityObject identityObject = iterator.next();
         groups.add(createGroup(identityObject));
      }

      return groups;
   }

   public Collection<Group> findGroup(GroupType groupType) throws IdentityException
   {
      return findGroup(groupType, null);
   }

   public boolean isVirtual(Identity identity)
   {
      //TODO:NYI
      throw new NotYetImplementedException("Postponed");
   }
}
