/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session.managers;

import org.jboss.identity.idm.api.RoleManager;
import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.RoleType;
import org.jboss.identity.idm.api.Role;
import org.jboss.identity.idm.api.Identity;
import org.jboss.identity.idm.api.Group;
import org.jboss.identity.idm.api.IdentityType;
import org.jboss.identity.idm.api.RoleManagerFeaturesDescription;
import org.jboss.identity.idm.api.IdentitySearchControl;
import org.jboss.identity.idm.api.GroupType;
import org.jboss.identity.idm.exception.IdentityException;
import org.jboss.identity.idm.spi.model.IdentityObjectRelationshipType;
import org.jboss.identity.idm.spi.model.IdentityObjectRelationship;
import org.jboss.identity.idm.spi.exception.OperationNotSupportedException;
import org.jboss.identity.idm.spi.searchcontrol.IdentityObjectSearchControl;
import org.jboss.identity.idm.impl.NotYetImplementedException;
import org.jboss.identity.idm.impl.api.session.managers.AbstractManager;
import org.jboss.identity.idm.impl.api.model.SimpleRoleType;
import org.jboss.identity.idm.impl.api.model.SimpleRole;

import java.util.Collection;
import java.util.Set;
import java.util.HashSet;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class RoleManagerImpl extends AbstractManager implements RoleManager
{
   public static final IdentityObjectRelationshipType ROLE = new IdentityObjectRelationshipType()
   {
      public String getName()
      {
         return "JBOSS_IDENTITY_ROLE";
      }
   };

   private final RoleManagerFeaturesDescription featuresDescription;

   public RoleManagerImpl(IdentitySession session)
   {
      super(session);

      featuresDescription = new RoleManagerFeaturesDescription()
      {
         public boolean isRoleTypeAddRemoveSupported()
         {
            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().isRelationshipNameAddRemoveSupported();
         }

         public boolean isRoleTypeSearchControlSupported(Class controlClazz)
         {
            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().isRoleNameControlSupported(controlClazz);
         }

         public boolean isRoleTypeSearchControlSupported(IdentitySearchControl control)
         {
            if (control instanceof IdentityObjectSearchControl)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isRoleNameControlSupported((IdentityObjectSearchControl)control);
            }
            else
            {
               return false;
            }
         }
      };
   }

   public RoleManagerFeaturesDescription getSupportedFeatures()
   {
      return featuresDescription;
   }

   public RoleType createRoleType(String name) throws IdentityException
   {
      String roleType = null;

      try
      {
         roleType = getRepository().createRelationshipName(getInvocationContext(), name);
      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }

      return new SimpleRoleType(roleType);
   }

   public void removeRoleType(String name) throws IdentityException
   {
      try
      {
         getRepository().removeRelationshipName(getInvocationContext(), name);
      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }
   }

   public void removeRoleType(RoleType roleType) throws IdentityException
   {
      removeRoleType(roleType.getName());
   }

   public RoleType getRoleType(String name) throws IdentityException
   {
      try
      {
         Set<String> names = getRepository().getRelationshipNames(getInvocationContext(), null);
         if (names.contains(name))
         {
            return new SimpleRoleType(name);
         }
      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }

      return null;
   }

   public Collection<RoleType> findRoleTypes(IdentitySearchControl[] controls) throws IdentityException
   {
      
      try
      {
         Set<String> names = getRepository().getRelationshipNames(getInvocationContext(), convertSearchControls(controls));
         Set<RoleType> types = new HashSet<RoleType>();

         for (String name : names)
         {
            types.add(new SimpleRoleType(name));
         }

         return types;
      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }

   }

   public Role createRole(RoleType roleType, Identity identity, Group group) throws IdentityException
   {
      //TODO: add createRoleType switch to the API

      IdentityObjectRelationship rel = getRepository().createRelationship(getInvocationContext(), createIdentityObject(identity), createIdentityObject(group), ROLE, roleType.getName(), false);

      //TODO: null id - IdentityObjectRelationship doesn't have id
      return new SimpleRole(new SimpleRoleType(rel.getName()), createIdentity(rel.getFromIdentityObject()), createGroup(rel.getToIdentityObject()));

   }

   public void removeRole(RoleType roleType, Identity identity, Group group) throws IdentityException
   {
      getRepository().removeRelationship(getInvocationContext(), createIdentityObject(identity), createIdentityObject(group), ROLE, roleType.getName());
   }

   public void removeRole(Role role) throws IdentityException
   {
      getRepository().removeRelationship(getInvocationContext(), createIdentityObject(role.getIdentity()), createIdentityObject(role.getGroup()), ROLE, role.getRoleType().getName());
   }

   public boolean hasRole(Identity identity, Group group, RoleType roleType) throws IdentityException
   {
      //TODO: does separate hasRelationship method in IdentityStore makes sense?

      Set<IdentityObjectRelationship> rels = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(identity), createIdentityObject(group), ROLE);

      for (IdentityObjectRelationship rel : rels)
      {
         if (rel.getType().getName().equals(ROLE.getName()) && rel.getName() != null && rel.getName().equals(roleType.getName()))
         {
            return true;
         }
      }

      return false;
   }

   public Collection<RoleType> findRoleTypes(Identity identity, Group group) throws IdentityException
   {
      return findRoleTypes(identity, group, null);
   }

   public Collection<RoleType> findRoleTypes(Identity identity, Group group, IdentitySearchControl[] controls) throws IdentityException
   {

      Set<IdentityObjectRelationship> rels = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(identity), createIdentityObject(group), ROLE);
      Set<RoleType> types = new HashSet<RoleType>();

      for (IdentityObjectRelationship rel : rels)
      {
         types.add(new SimpleRoleType(rel.getName()));
      }

      return types;


   }

   public Collection<RoleType> findIdentityRoleTypes(Identity identity) throws IdentityException
   {
      return findIdentityRoleTypes(identity, null);
   }

   public Collection<RoleType> findIdentityRoleTypes(Identity identity, IdentitySearchControl[] controls) throws IdentityException
   {
      Set<RoleType> types = new HashSet<RoleType>();

      try
      {
         Collection<String> names = getRepository().getRelationshipNames(getInvocationContext(), createIdentityObject(identity), convertSearchControls(controls));

         for (String name : names)
         {
            types.add(new SimpleRoleType(name));
         }

         return types;

      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported", e);
      }

   }

   public Collection<RoleType> findGroupRoleTypes(Group group) throws IdentityException
   {
      return findGroupRoleTypes(group, null);
   }

   public Collection<RoleType> findGroupRoleTypes(Group group, IdentitySearchControl[] controls) throws IdentityException
   {
      Set<RoleType> types = new HashSet<RoleType>();

      try
      {
         Collection<String> names = getRepository().getRelationshipNames(getInvocationContext(), createIdentityObject(group), convertSearchControls(controls));

         for (String name : names)
         {
            types.add(new SimpleRoleType(name));
         }

         return types;

      }
      catch (OperationNotSupportedException e)
      {
         throw new IdentityException("Role management not supported");
      }

   }

   public Collection<Role> findRoles(IdentityType identityType, RoleType roleType) throws IdentityException
   {
      Set<Role> roles = new HashSet<Role>();

      Set<IdentityObjectRelationship> relationships = null;

      // If Identity then search for parent relationships
      if (identityType instanceof Identity)
      {
         relationships = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(identityType), ROLE, true, true, null);
      }
      // If Group then search for child relationships
      else
      {
         relationships = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(identityType), ROLE, false, true, null);
      }

      for (IdentityObjectRelationship relationship : relationships)
      {
         if (roleType.getName().equals(relationship.getName()))
         {
            roles.add(new SimpleRole(new SimpleRoleType(relationship.getName()), createIdentity(relationship.getFromIdentityObject()), createGroup(relationship.getToIdentityObject())));
         }
      }

      return roles;

   }
}
