/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.repository;

import org.jboss.identity.idm.spi.repository.IdentityStoreRepository;
import org.jboss.identity.idm.spi.store.IdentityStore;
import org.jboss.identity.idm.spi.store.AttributeStore;
import org.jboss.identity.idm.spi.store.IdentityStoreInvocationContext;
import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityRepositoryConfigurationMetaData;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityStoreMappingMetaData;
import org.jboss.identity.idm.exception.IdentityException;

import java.util.Set;
import java.util.Map;
import java.util.HashSet;
import java.util.List;
import java.util.HashMap;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public abstract class AbstractIdentityStoreRepository implements IdentityStoreRepository
{

   protected Map<String, IdentityStore> identityStoreMappings = new HashMap<String, IdentityStore>();

   protected Map<String, AttributeStore> attributeStoreMappings = new HashMap<String, AttributeStore>();

   protected IdentityStore defaultIdentityStore;

   protected AttributeStore defaultAttributeStore;
   

   public void bootstrap(IdentityRepositoryConfigurationMetaData configurationMD,
                         Map<String, IdentityStore> bootstrappedIdentityStores,
                         Map<String, AttributeStore> bootstrappedAttributeStores) throws IdentityException
   {

      String asId = configurationMD.getDefaultAttributeStoreId();
      String isId = configurationMD.getDefaultIdentityStoreId();

      if (asId != null && bootstrappedAttributeStores.keySet().contains(asId))
      {
         defaultAttributeStore = bootstrappedAttributeStores.get(asId);
      }

      if (isId != null && bootstrappedIdentityStores.keySet().contains(isId))
      {
         defaultIdentityStore = bootstrappedIdentityStores.get(isId);
      }

      for (IdentityStoreMappingMetaData identityStoreMappingMetaData : configurationMD.getIdentityStoreToIdentityObjectTypeMappings())
      {
         String storeId = identityStoreMappingMetaData.getIdentityStoreId();
         List<String> identityObjectTypeMappings = identityStoreMappingMetaData.getIdentityObjectTypeMappings();

         IdentityStore store = bootstrappedIdentityStores.get(storeId);

         if (store == null)
         {
            throw new IdentityException("Mapped IdentityStore not available: " + storeId);
         }

         for (String mapping : identityObjectTypeMappings)
         {
            identityStoreMappings.put(mapping, store);
            attributeStoreMappings.put(mapping, store);
         }

      }

   }

   public Set<IdentityStore> getConfiguredIdentityStores()
   {
      return new HashSet<IdentityStore>(identityStoreMappings.values());
   }

   public Set<AttributeStore> getConfiguredAttributeStores()
   {
      return new HashSet<AttributeStore>(attributeStoreMappings.values());
   }

   public Map<String, IdentityStore> getIdentityStoreMappings()
   {
      return identityStoreMappings;
   }

   public Map<String, AttributeStore> getAttributeStoreMappings()
   {
      return attributeStoreMappings;
   }

   public IdentityStore getIdentityStore(IdentityObjectType identityObjectType)
   {
      return identityStoreMappings.get(identityObjectType.getName());
   }

   public AttributeStore getAttributeStore(IdentityObjectType identityObjectType)
   {
      return attributeStoreMappings.get(identityObjectType.getName());
   }

   protected boolean hasIdentityObject(IdentityStoreInvocationContext ctx, IdentityStore is, IdentityObject io) throws IdentityException
   {
      try
      {
         if (is.findIdentityObject(ctx, io.getName(), io.getIdentityType()) != null)
         {
            return true;
         }
      }
      catch (IdentityException e)
      {
         //store may throw exception if there is no identity so do nothing
         //TODO: should have isIdentityPresent method
         
      }
      return false;
   }
}
