/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session.managers;

import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.AttributesManager;
import org.jboss.identity.idm.api.IdentityType;
import org.jboss.identity.idm.api.Identity;
import org.jboss.identity.idm.api.AttributeDescription;
import org.jboss.identity.idm.api.CredentialType;
import org.jboss.identity.idm.api.Credential;
import org.jboss.identity.idm.api.Attribute;
import org.jboss.identity.idm.exception.IdentityException;
import org.jboss.identity.idm.spi.model.IdentityObjectCredential;
import org.jboss.identity.idm.spi.model.IdentityObjectAttribute;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityObjectAttributeMetaData;
import org.jboss.identity.idm.impl.api.session.managers.AbstractManager;
import org.jboss.identity.idm.impl.api.attribute.IdentityObjectAttributeMetaDataImpl;
import org.jboss.identity.idm.impl.api.SimpleAttribute;
import org.jboss.identity.idm.impl.api.PasswordCredential;
import org.jboss.identity.idm.impl.api.SimpleCredentialType;

import java.util.Set;
import java.util.Map;
import java.util.HashMap;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class AttributesManagerImpl extends AbstractManager implements AttributesManager
{

   public AttributesManagerImpl(IdentitySession session)
   {
      super(session);
   }

   public AttributeDescription getAttributeDescription(IdentityType identityType, String name)
   {

      Map<String, IdentityObjectAttributeMetaData> mdMap =
         getRepository().getAttributesMetaData(getInvocationContext(), createIdentityObject(identityType).getIdentityType());

      if (mdMap != null && mdMap.containsKey(name))
      {
         IdentityObjectAttributeMetaData attributeMD = mdMap.get(name);
         if (attributeMD instanceof AttributeDescription)
         {
            return (AttributeDescription)attributeMD;
         }
         else
         {
            return new IdentityObjectAttributeMetaDataImpl(attributeMD);
         }
      }
      
      return null;

   }

   public Map<String, AttributeDescription> getSupportedAttributesDescriptions(IdentityType identityType)
   {

      Map<String, IdentityObjectAttributeMetaData> mdMap =
         getRepository().getAttributesMetaData(getInvocationContext(), createIdentityObject(identityType).getIdentityType());

      Map<String, AttributeDescription> descriptionMap = new HashMap<String, AttributeDescription>();

      if (mdMap != null)
      {
         for (IdentityObjectAttributeMetaData attributeMD : mdMap.values())
         {
            if (attributeMD instanceof AttributeDescription)
            {
               descriptionMap.put(attributeMD.getName(), (AttributeDescription)attributeMD);
            }
            else
            {
               descriptionMap.put(attributeMD.getName(), new IdentityObjectAttributeMetaDataImpl(attributeMD));
            }
         }
      }

      return descriptionMap;
   }

   public Set<String> getSupportedAttributeNames(IdentityType identityType) throws IdentityException
   {
      return getRepository().getSupportedAttributeNames(getInvocationContext(), createIdentityObject(identityType).getIdentityType());
   }

   public Map<String, Attribute> getAttributes(IdentityType identity) throws IdentityException
   {
      Map<String, IdentityObjectAttribute> map = getRepository().getAttributes(getInvocationContext(), createIdentityObject(identity));

      Map<String, Attribute> newMap = new HashMap<String, Attribute>();

      for (Map.Entry<String, IdentityObjectAttribute> entry : map.entrySet())
      {
         newMap.put(entry.getKey(), convertAttribute(entry.getValue()));
      }
      return newMap;
   }

   public void updateAttributes(IdentityType identity, Attribute[] attributes) throws IdentityException
   {
      getRepository().updateAttributes(getInvocationContext(), createIdentityObject(identity), convertAttributes(attributes));

   }

   public Attribute getAttribute(IdentityType identity, String attributeName) throws IdentityException
   {
      return getAttributes(identity).get(attributeName);
   }

   public void addAttribute(IdentityType identity, String attributeName, Object[] values) throws IdentityException
   {
      Attribute[] attrs = new Attribute[]{new SimpleAttribute(attributeName, values)};

      addAttributes(identity, attrs);

   }

   public void addAttribute(IdentityType identity, String attributeName, Object value) throws IdentityException
   {
      Attribute[] attrs = new Attribute[]{new SimpleAttribute(attributeName, value)};


      addAttributes(identity, attrs);

   }

   public void addAttributes(IdentityType identity, Attribute[] attributes) throws IdentityException
   {
      getRepository().addAttributes(getInvocationContext(), createIdentityObject(identity), convertAttributes(attributes));
   }

   public void removeAttributes(IdentityType identity, String[] attributeNames) throws IdentityException
   {
      getRepository().removeAttributes(getInvocationContext(), createIdentityObject(identity), attributeNames);
   }

   public boolean hasPassword(Identity identity) throws IdentityException
   {
      return getRepository().getSupportedFeatures().isCredentialSupported(createIdentityObject(identity).getIdentityType(), PasswordCredential.TYPE);
   }

   public boolean validatePassword(Identity identity, String password) throws IdentityException
   {
      return getRepository().validateCredential(getInvocationContext(), createIdentityObject(identity), new PasswordCredential(password));
   }

   public void updatePassword(Identity identity, String password) throws IdentityException
   {
      getRepository().updateCredential(getInvocationContext(), createIdentityObject(identity), new PasswordCredential(password));
   }

   public boolean hasCredential(Identity identity, CredentialType credentialType) throws IdentityException
   {
      return getRepository().getSupportedFeatures().isCredentialSupported(createIdentityObject(identity).getIdentityType(), new SimpleCredentialType(credentialType.getName()));
   }

   public boolean validateCredentials(Identity identity, Credential[] credentials) throws IdentityException
   {

      for (Credential credential : credentials)
      {
         IdentityObjectCredential ioc = null;

         //Handle only those credentials that implement SPI

         if (!(credential instanceof IdentityObjectCredential))
         {
            throw new IdentityException("Unsupported Credential implementation: " + credential.getClass());
         }

         // All credentials must pass

         if (!getRepository().validateCredential(getInvocationContext(), createIdentityObject(identity), ioc))
         {
            return false;
         }
      }

      return true;
   }

   public void updateCredential(Identity identity, Credential credential) throws IdentityException
   {
      if (credential instanceof IdentityObjectCredential)
      {
         getRepository().updateCredential(getInvocationContext(), createIdentityObject(identity), (IdentityObjectCredential)credential);
      }
      else
      {
         throw new IdentityException("Unsupported Credential implementation: " + credential.getClass());
      }
   }
}
