/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session.managers;

import org.jboss.identity.idm.api.RelationshipManager;
import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.Group;
import org.jboss.identity.idm.api.Identity;
import org.jboss.identity.idm.api.IdentityType;
import org.jboss.identity.idm.api.GroupType;
import org.jboss.identity.idm.api.RelationshipManagerFeaturesDescription;
import org.jboss.identity.idm.api.IdentitySearchControl;
import org.jboss.identity.idm.exception.IdentityException;
import org.jboss.identity.idm.spi.model.IdentityObjectRelationshipType;
import org.jboss.identity.idm.spi.model.IdentityObjectRelationship;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.searchcontrol.IdentityObjectSearchControl;
import org.jboss.identity.idm.impl.NotYetImplementedException;
import org.jboss.identity.idm.impl.api.session.managers.AbstractManager;
import org.jboss.identity.idm.impl.api.session.mapper.IdentityObjectTypeMapper;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.LinkedList;
import java.util.Set;
import java.util.HashSet;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class RelationshipManagerImpl extends AbstractManager implements RelationshipManager
{
   RelationshipManagerFeaturesDescription featuresDescription;

   public static final IdentityObjectRelationshipType MEMBER = new IdentityObjectRelationshipType()
   {
      public String getName()
      {
         return "JBOSS_IDENTITY_MEMBERSHIP";
      }
   };

   public RelationshipManagerImpl(IdentitySession session)
   {
      super(session);

      featuresDescription = new RelationshipManagerFeaturesDescription()
      {
         public boolean isIdentityAssociationSupported(GroupType fromGroupType)
         {

            IdentityObjectType identityOT = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();
            IdentityObjectType groupIdentityOT = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(fromGroupType);

            try
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isRelationshipTypeSupported(groupIdentityOT, identityOT, MEMBER);
            }
            catch (IdentityException e)
            {
               return false;
            }

         }

         public boolean isGroupAssociationSupported(GroupType fromGroupType, GroupType toGroupType)
         {
            IdentityObjectType toGroupOT = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(toGroupType);
            IdentityObjectType fromGroupOT = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(fromGroupType);

            try
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isRelationshipTypeSupported(fromGroupOT, toGroupOT, MEMBER);
            }
            catch (IdentityException e)
            {
               return false;
            }
         }

         public boolean isIdentitiesSearchControlSupported(IdentitySearchControl control)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

            if (control instanceof IdentityObjectSearchControl)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isControlSupported(objectType, (IdentityObjectSearchControl)control);
            }
            
            return false;
         }


         public boolean isIdentitiesSearchControlSupported(Class controlClazz)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
               isControlSupported(objectType, controlClazz);

         }

         public boolean isGroupsSearchControlSupported(GroupType groupType, IdentitySearchControl control)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);

            if (control instanceof IdentityObjectSearchControl)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isControlSupported(objectType, (IdentityObjectSearchControl)control);
            }

            return false;
         }

         public boolean isGroupsSearchControlSupported(GroupType groupType, Class controlClazz)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);

            return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
               isControlSupported(objectType, controlClazz);
         }
      };
   }

   public RelationshipManagerFeaturesDescription getFeaturesDescription()
   {
      return featuresDescription;
   }

   public void associateGroups(Collection<Group> parents, Collection<Group> members) throws IdentityException
   {
      for (Iterator<Group> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         Group parent = parentsIterator.next();

         for (Iterator<Group> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            Group member = membersIterator.next();

            associateGroups(parent, member);
         }
      }
   }

   public void associateGroups(Group parent, Group member) throws IdentityException
   {
      getRepository().createRelationship(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER, null, true);
   }

   public void associateIdentities(Collection<Group> parents, Collection<Identity> members) throws IdentityException
   {
      for (Iterator<Group> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         Group parent = parentsIterator.next();

         for (Iterator<Identity> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            Identity member = membersIterator.next();

            associateIdentities(parent, member);
         }
      }
   }

   public void associateIdentities(Group parent, Identity member) throws IdentityException
   {
      getRepository().createRelationship(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER, null, true);
   }

   public void disassociateGroups(Collection<Group> parents, Collection<Group> members) throws IdentityException
   {
      for (Iterator<Group> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         Group parent = parentsIterator.next();

         for (Iterator<Group> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            Group member = membersIterator.next();

            getRepository().removeRelationship(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER, null);
         }
      }
   }

   public void disassociateIdentities(Collection<Group> parents, Collection<Identity> members) throws IdentityException
   {
      for (Iterator<Group> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         Group parent = parentsIterator.next();

         for (Iterator<Identity> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            Identity member = membersIterator.next();

            getRepository().removeRelationship(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER, null);
         }
      }

   }

   public <G extends IdentityType, I extends IdentityType> boolean isAssociated(Collection<G> parents, Collection<I> members) throws IdentityException
   {
      //TODO: maybe IdentityStore should have isRelationshipPresent method to improve this?

      for (Iterator<G> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         IdentityType parent = parentsIterator.next();

         for (Iterator<I> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            IdentityType member = membersIterator.next();

            Collection<IdentityObjectRelationship> relationships = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER);

            if (relationships.size() == 0)
            {
               return false;
            }
         }
      }

      return true;
   }

   public <G extends IdentityType, I extends IdentityType> boolean isAssociated(G parent, I member) throws IdentityException
   {
      Set<G> parents = new HashSet<G>();
      parents.add(parent);
      Set<I> members = new HashSet<I>();
      members.add(member);

      return isAssociated(parents, members);
   }

   public Collection<Group> findAssociatedGroups(Group group, GroupType groupType, boolean parent, boolean inherited, IdentitySearchControl[] controls) throws IdentityException
   {

      List<Group> identities = new LinkedList<Group>();

      IdentityObjectType iot = getIdentityObjectType(groupType);

      //TODO Handle inherited
      if (inherited)
      {
         throw new NotYetImplementedException("Support for 'inherited' argument is not yet implemented. Please use 'false' value for now");
      }

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(group), MEMBER, parent, convertSearchControls(controls));

      for (IdentityObject io : ios)
      {
         if (io.getIdentityType().getName().equals(iot.getName()))
         {
            identities.add(createGroup(io));
         }
      }

      return identities;

   }

   public Collection<Group> findAssociatedGroups(Group group, GroupType groupType, boolean parent, boolean inherited) throws IdentityException
   {
      return findAssociatedGroups(group, groupType, parent, inherited, null);
   }

   public Collection<Group> findAssociatedGroups(Identity identity, GroupType groupType, IdentitySearchControl[] controls) throws IdentityException
   {
      List<Group> identities = new LinkedList<Group>();

      IdentityObjectType iot = getIdentityObjectType(groupType);

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(identity), MEMBER, false, convertSearchControls(controls));

      for (IdentityObject io : ios)
      {
         if (io.getIdentityType().getName().equals(iot.getName()))
         {
            identities.add(createGroup(io));
         }
      }

      return identities;
   }

   public Collection<Group> findAssociatedGroups(Identity identity, GroupType groupType) throws IdentityException
   {
      return findAssociatedGroups(identity, groupType, null);
   }

   public Collection<Group> findAssociatedGroups(Identity identity, IdentitySearchControl[] controls) throws IdentityException
   {
      List<Group> identities = new LinkedList<Group>();

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(identity), MEMBER, false, convertSearchControls(controls));

      for (IdentityObject io : ios)
      {
            identities.add(createGroup(io));
      }

      return identities;
   }

   public Collection<Group> findAssociatedGroups(Identity identity) throws IdentityException
   {
      return findAssociatedGroups(identity, new IdentitySearchControl[] {});
   }

   public Collection<Identity> findAssociatedIdentities(Group group, boolean inherited, IdentitySearchControl[] controls) throws IdentityException
   {
      List<Identity> identities = new LinkedList<Identity>();

      //TODO Handle inherited
      if (inherited)
      {
         throw new NotYetImplementedException("Support for 'inherited' argument is not yet implemented. Please use 'false' value for now");
      }

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(group), MEMBER, true, convertSearchControls(controls));

      //TODO: filter out groups....

      for (IdentityObject io : ios)
      {
         identities.add(createIdentity(io));
      }

      return identities;
   }

   public Collection<Identity> findAssociatedIdentities(Group group, boolean inherited) throws IdentityException
   {
      return findAssociatedIdentities(group, inherited, null);
   }
}
