/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.store.ldap;

import org.jboss.identity.idm.spi.configuration.metadata.IdentityObjectTypeMetaData;
import org.jboss.identity.idm.spi.configuration.metadata.RelationshipMetaData;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityObjectAttributeMetaData;

import java.util.Map;
import java.util.Set;
import java.util.Collections;
import java.util.List;
import java.util.LinkedList;
import java.util.HashMap;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class SimpleLDAPIdentityObjectTypeConfiguration implements LDAPIdentityObjectTypeConfiguration
{
   private final String idAttributeName;

   private final String passwordAttributeName;

   private final String[] ctxDNs;

   private final String entrySearchFilter;

   private final boolean allowCreateEntry;

   private final Map<String, String[]> createEntryAttributeValues;

   private final String[] allowedMembershipTypes;

   private final String membershipAttributeName;

   private final boolean isMembershipAttributeDN;

   private final boolean allowEmptyMemberships;

   private final Map<String, String> attributeNames;


   //Consts

   public static final String ID_ATTRIBUTE_NAME = "idAttributeName";

   public static final String PASSWORD_ATTRIBUTE_NAME = "passwordAttributeName";

   public static final String CTX_DNS = "ctxDNs";

   public static final String ENTRY_SEARCH_FILTER = "entrySearchFilter";

   public static final String ALLOW_CREATE_ENTRY = "allowCreateEntry";

   public static final String MEMBERSHIP_ATTRIBUTE_NAME = "membershipAttributeName";

   public static final String IS_MEMBERSHIP_ATTRIBUTE_DN = "isMembershipAttributeDN";

   public static final String ALLOW_EMPTY_MEMBERSHIPS = "allowEmptyMemberships";

   public static final String CREATE_ENTRY_ATTRIBUTE_VALUES = "createEntryAttributeValues";



   public SimpleLDAPIdentityObjectTypeConfiguration(IdentityObjectTypeMetaData objectTypeMD)
   {
      this.idAttributeName = objectTypeMD.getOptionSingleValue(ID_ATTRIBUTE_NAME);
      this.passwordAttributeName = objectTypeMD.getOptionSingleValue(PASSWORD_ATTRIBUTE_NAME);
      this.entrySearchFilter = objectTypeMD.getOptionSingleValue(ENTRY_SEARCH_FILTER);
      this.membershipAttributeName = objectTypeMD.getOptionSingleValue(MEMBERSHIP_ATTRIBUTE_NAME);
      String allowCreateEntry = objectTypeMD.getOptionSingleValue(ALLOW_CREATE_ENTRY);
      if (allowCreateEntry != null && allowCreateEntry.equalsIgnoreCase("true"))
      {
         this.allowCreateEntry = true;
      }
      else
      {
         this.allowCreateEntry = false;
      }

      String isMembershipAttributeDN = objectTypeMD.getOptionSingleValue(IS_MEMBERSHIP_ATTRIBUTE_DN);
      if (isMembershipAttributeDN != null && isMembershipAttributeDN.equalsIgnoreCase("true"))
      {
         this.isMembershipAttributeDN = true;
      }
      else
      {
         this.isMembershipAttributeDN = false;
      }

      String allowEmptyMemberships = objectTypeMD.getOptionSingleValue(ALLOW_EMPTY_MEMBERSHIPS);
      if (allowEmptyMemberships != null && allowEmptyMemberships.equalsIgnoreCase("true"))
      {
         this.allowEmptyMemberships = true;
      }
      else
      {
         this.allowEmptyMemberships = false;
      }

      List<String> relationships = new LinkedList<String>();

      if (objectTypeMD.getRelationships() != null)
      {
         for (RelationshipMetaData relationshipMetaData : objectTypeMD.getRelationships())
         {
            relationships.add(relationshipMetaData.getIdentityObjectTypeRef());
         }
      }

      allowedMembershipTypes = relationships.toArray(new String[relationships.size()]);
      
      Map<String, String> attrsNames = new HashMap<String, String>();

      for (IdentityObjectAttributeMetaData attributeMetaData : objectTypeMD.getAttributes())
      {
         attrsNames.put(attributeMetaData.getName(), attributeMetaData.getStoreMapping());
      }

      attributeNames = Collections.unmodifiableMap(attrsNames);

      List<String> dns = objectTypeMD.getOption(CTX_DNS);
      if (dns != null)
      {
         this.ctxDNs = dns.toArray(new String[dns.size()]);
      }
      else
      {
         this.ctxDNs = null;
      }

      Map<String, List<String>> createEntryAttributesMap = new HashMap<String, List<String>>();

      List<String> createAttributes = objectTypeMD.getOption(CREATE_ENTRY_ATTRIBUTE_VALUES);

      if (createAttributes != null && createAttributes.size() > 0 )
      {
         for (String attribute : createAttributes)
         {
            String[] parts = attribute.split("=", 2);
            if (parts.length != 2)
            {
               continue;
            }

            String name = parts[0];
            String value = parts[1];

            if (!createEntryAttributesMap.containsKey(name))
            {
               List<String> list = new LinkedList<String>();
               list.add(value);
               createEntryAttributesMap.put(name, list);
            }
            else
            {
               createEntryAttributesMap.get(name).add(value);
            }
         }

         Map<String, String[]> createEntryAttributesArray = new HashMap<String, String[]>();

         for (Map.Entry<String, List<String>> entry : createEntryAttributesMap.entrySet())
         {
            createEntryAttributesArray.put(entry.getKey(), entry.getValue().toArray(new String[entry.getValue().size()]));
         }

         this.createEntryAttributeValues = Collections.unmodifiableMap(createEntryAttributesArray);
      }
      else
      {
         this.createEntryAttributeValues = Collections.unmodifiableMap(new HashMap<String, String[]>());
      }

      //TODO: validate all required options - throw exception for missing ones and set defaults for others



   }

   public SimpleLDAPIdentityObjectTypeConfiguration(String idAttributeName,
                                                    String passwordAttributeName,
                                                    String[] ctxDNs,
                                                    String entrySearchFilter,
                                                    boolean allowCreateEntry,
                                                    Map<String, String[]> createEntryAttributeValues,
                                                    String[] allowedMembershipTypes,
                                                    String membershipAttributeName,
                                                    boolean membershipAttributeDN,
                                                    boolean allowEmptyMemberships,
                                                    Map<String, String> attributeNames)
   {
      this.idAttributeName = idAttributeName;
      this.passwordAttributeName = passwordAttributeName;
      this.ctxDNs = ctxDNs.clone();
      this.entrySearchFilter = entrySearchFilter;
      this.allowCreateEntry = allowCreateEntry;
      this.createEntryAttributeValues = Collections.unmodifiableMap(createEntryAttributeValues);
      this.allowedMembershipTypes = allowedMembershipTypes.clone();
      this.membershipAttributeName = membershipAttributeName;
      isMembershipAttributeDN = membershipAttributeDN;
      this.allowEmptyMemberships = allowEmptyMemberships;
      this.attributeNames = Collections.unmodifiableMap(attributeNames);
   }



   public String getIdAttributeName()
   {
      return idAttributeName;
   }

   public String[] getCtxDNs()
   {
      return ctxDNs.clone();
   }

   public String getEntrySearchFilter()
   {
      return entrySearchFilter;
   }

   public boolean isAllowCreateEntry()
   {
      return allowCreateEntry;
   }

   public Map<String, String[]> getCreateEntryAttributeValues()
   {
      return createEntryAttributeValues;
   }

   public String[] getAllowedMembershipTypes()
   {
      return allowedMembershipTypes.clone();
   }

   public String getMembershipAttributeName()
   {
      return membershipAttributeName;
   }

   public boolean isMembershipAttributeDN()
   {
      return isMembershipAttributeDN;
   }

   public boolean allowEmptyMemberships()
   {
      return allowEmptyMemberships;
   }

   public String getAttributeMapping(String name)
   {
      return attributeNames.get(name);
   }

   public String getPasswordAttributeName()
   {
      return passwordAttributeName;
   }

   //   public void setIdAttributeName(String idAttributeName)
//   {
//      this.idAttributeName = idAttributeName;
//   }
//
//   public void setCtxDNs(String[] ctxDNs)
//   {
//      this.ctxDNs = ctxDNs;
//   }
//
//   public void setEntrySearchFilter(String entrySearchFilter)
//   {
//      this.entrySearchFilter = entrySearchFilter;
//   }
//
//   public void setAllowCreateEntry(boolean allowCreateEntry)
//   {
//      this.allowCreateEntry = allowCreateEntry;
//   }
//
//   public void setCreateEntryAttributeValues(Map<String, String[]> createEntryAttributeValues)
//   {
//      this.createEntryAttributeValues = createEntryAttributeValues;
//   }
//
//   public void setAllowedMembershipTypes(String[] allowedMembershipTypes)
//   {
//      this.allowedMembershipTypes = allowedMembershipTypes;
//   }
//
//   public void setMembershipAttributeName(String membershipAttributeName)
//   {
//      this.membershipAttributeName = membershipAttributeName;
//   }
//
//   public void setMembershipAttributeDN(boolean membershipAttributeDN)
//   {
//      isMembershipAttributeDN = membershipAttributeDN;
//   }
//
//   public void setAllowEmptyMemberships(boolean allowEmptyMemberships)
//   {
//      this.allowEmptyMemberships = allowEmptyMemberships;
//   }
//
//   public void setAttributeNames(Map<String, String> attributeNames)
//   {
//      this.attributeNames = attributeNames;
//   }

   public Set<String> getMappedAttributesNames()
   {
      return Collections.unmodifiableSet(attributeNames.keySet());
   }
}
