/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.store.ldap;

import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityStoreConfigurationMetaData;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityObjectTypeMetaData;
import org.jboss.identity.idm.impl.types.SimpleIdentityObjectType;

import java.util.Map;
import java.util.HashMap;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class SimpleLDAPIdentityStoreConfiguration implements LDAPIdentityStoreConfiguration
{
   private final String providerURL;

   private final String adminDN;

   private final String adminPassword;

   private final int searchTimeLimit;

   private final Map<String, LDAPIdentityObjectTypeConfiguration> typesConfiguration;


   // Consts

   public static final String PROVIDER_URL = "providerURL";

   public static final String ADMIN_DN = "adminDN";

   public static final String ADMIN_PASSWORD = "adminPassword";

   public static final String SEARCH_TIME_LIMIT = "searchTimeLimit";

   public static final int SEARCH_TIME_LIMIT_DEFAULT = 10000;

   public SimpleLDAPIdentityStoreConfiguration(IdentityStoreConfigurationMetaData storeMD)
   {
      if (storeMD == null)
      {
         throw new IllegalArgumentException();
      }

      providerURL = storeMD.getOptionSingleValue(PROVIDER_URL);
      adminDN = storeMD.getOptionSingleValue(ADMIN_DN);
      adminPassword = storeMD.getOptionSingleValue(ADMIN_PASSWORD);
      String searchTL = storeMD.getOptionSingleValue(SEARCH_TIME_LIMIT);

      if (searchTL != null)
      {
         searchTimeLimit = Integer.valueOf(searchTL);
      }
      else
      {
         searchTimeLimit = SEARCH_TIME_LIMIT_DEFAULT;
      }

      Map<String, LDAPIdentityObjectTypeConfiguration> types = new HashMap<String, LDAPIdentityObjectTypeConfiguration>();

      for (IdentityObjectTypeMetaData identityObjectTypeMetaData : storeMD.getSupportedIdentityTypes())
      {
         LDAPIdentityObjectTypeConfiguration typeConfig = new SimpleLDAPIdentityObjectTypeConfiguration(identityObjectTypeMetaData);
         types.put(identityObjectTypeMetaData.getName(), typeConfig);

      }

      typesConfiguration = types;

      //TODO: validate if critical values are present


   }

   public String getProviderURL()
   {
      return providerURL;
   }

   public String getAdminDN()
   {
      return adminDN;
   }

   public String getAdminPassword()
   {
      return adminPassword;
   }

   public int getSearchTimeLimit()
   {
      return searchTimeLimit;
   }

   public LDAPIdentityObjectTypeConfiguration getTypeConfiguration(String typeName)
   {
      return typesConfiguration.get(typeName);
   }

   public IdentityObjectType[] getConfiguredTypes()
   {
      IdentityObjectType[] types = new IdentityObjectType[typesConfiguration.size()];
      Object[] names = typesConfiguration.keySet().toArray();

      for (int i = 0; i < names.length; i++)
      {
         String name = names[i].toString();
         types[i] = new SimpleIdentityObjectType(name);
      }

      return types;
   }
   
}
