/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session.managers;

import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.User;
import org.jboss.identity.idm.api.Group;
import org.jboss.identity.idm.api.IdentityType;
import org.jboss.identity.idm.api.Attribute;
import org.jboss.identity.idm.api.IdentitySearchCriteria;
import org.jboss.identity.idm.api.Role;
import org.jboss.identity.idm.spi.repository.IdentityStoreRepository;
import org.jboss.identity.idm.spi.store.IdentityStoreInvocationContext;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.model.IdentityObjectAttribute;
import org.jboss.identity.idm.spi.model.IdentityObjectRelationship;
import org.jboss.identity.idm.spi.search.IdentityObjectSearchCriteria;
import org.jboss.identity.idm.impl.types.SimpleIdentityObject;
import org.jboss.identity.idm.impl.types.SimpleIdentityObjectRelationship;
import org.jboss.identity.idm.impl.api.session.context.IdentitySessionContext;
import org.jboss.identity.idm.impl.api.session.IdentitySessionImpl;
import org.jboss.identity.idm.impl.api.model.SimpleUser;
import org.jboss.identity.idm.impl.api.model.SimpleGroup;
import org.jboss.identity.idm.impl.api.model.GroupId;
import org.jboss.identity.idm.impl.api.SimpleAttribute;
import org.jboss.identity.idm.impl.api.IdentitySearchCriteriaImpl;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public abstract class AbstractManager
{
   protected final IdentitySession identitySession;

   protected AbstractManager(IdentitySession session)
   {
      this.identitySession = session;
   }

   public IdentitySession getIdentitySession()
   {
      return identitySession;
   }

   public IdentitySearchCriteria createIdentitySearchConstraints()
   {
      return new IdentitySearchCriteriaImpl();
   }

   protected IdentitySessionContext getSessionContext()
   {
      if (identitySession instanceof IdentitySessionImpl)
      {
         return ((IdentitySessionImpl)identitySession).getSessionContext();
      }
      return null;
   }

   protected IdentityStoreRepository getRepository()
   {
      return getSessionContext().getIdentityStoreRepository();
   }

   protected IdentityStoreInvocationContext getInvocationContext()
   {
      return getSessionContext().resolveStoreInvocationContext();
   }

   protected User createUser(IdentityObject identityObject)
   {
      return new SimpleUser(identityObject.getName());
   }

   protected User createUserFromId(String name)
   {
      return new SimpleUser(name);
   }

   protected Group createGroup(IdentityObject identityObject)
   {
      String groupType = getSessionContext().getIdentityObjectTypeMapper().getGroupType(identityObject.getIdentityType());
      return new SimpleGroup(identityObject.getName(), groupType);
   }

   protected IdentityType createIdentityTypeFromId(String id)
   {
      if (GroupId.validateId(id))
      {
         GroupId groupId = new GroupId(id);

         return new SimpleGroup(groupId);
      }
      else
      {
         return new SimpleUser(id);
      }
   }

   protected IdentityObject createIdentityObject(User identity)
   {
      IdentityObjectType iot = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

      return new SimpleIdentityObject(identity.getId(), iot);
   }

   protected IdentityObject createIdentityObjectForUserName(String userName)
   {
      IdentityObjectType iot = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

      return new SimpleIdentityObject(userName, iot);
   }

   protected IdentityObject createIdentityObject(Group group)
   {
      IdentityObjectType iot = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(group.getGroupType());

      return new SimpleIdentityObject(group.getName(), group.getId(), iot);
   }

   protected IdentityObject createIdentityObjectForGroupId(String groupId)
   {
      GroupId id = new GroupId(groupId);

      IdentityObjectType iot = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(id.getType());

      return new SimpleIdentityObject(id.getName(), null, iot);
   }

   protected IdentityObject createIdentityObject(IdentityType identityType)
   {
      if (identityType instanceof User)
      {
         return createIdentityObject((User)identityType);
      }
      else if (identityType instanceof Group)
      {
         return createIdentityObject((Group)identityType);
      }

      throw new IllegalStateException("Not supported IdentityType extension: " + identityType.getClass());

   }

   protected IdentityObject createIdentityObject(String id)
   {
      if (GroupId.validateId(id))
      {
         GroupId groupId = new GroupId(id);

         return createIdentityObjectForGroupId(id);
      }
      else
      {
         return createIdentityObjectForUserName(id);
      }
   }



   protected Group createGroupFromId(String id)
   {
      return new SimpleGroup(new GroupId(id));
   }

   protected IdentityObjectSearchCriteria convertSearchControls(IdentitySearchCriteria controls)
   {
      if (controls == null)
      {
         return null;
      }

      if (controls instanceof IdentityObjectSearchCriteria)
      {
         return (IdentityObjectSearchCriteria)controls;
      }
      else
      {
         throw new IllegalArgumentException("Not supported IdentitySearchConstraints implementation: " + controls.getClass());
      }
   }

   protected IdentityObjectType getUserObjectType()
   {
      return getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();
   }

   protected IdentityObjectType getIdentityObjectType(String groupType)
   {
      return getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);
   }

   protected IdentityObjectAttribute[] convertAttributes(Attribute[] attributes)
   {
      IdentityObjectAttribute[] convertedAttributes = new IdentityObjectAttribute[attributes.length];

      for (int i = 0; i < attributes.length; i++)
      {
         convertedAttributes[i] = convertAttribute(attributes[i]);
      }
      return convertedAttributes;
   }

   protected Attribute[] convertAttributes(IdentityObjectAttribute[] attributes)
   {
      Attribute[] convertedAttributes = new Attribute[attributes.length];

      for (int i = 0; i < attributes.length; i++)
      {
         convertedAttributes[i] = convertAttribute(attributes[i]);
      }
      return convertedAttributes;
   }

   protected Attribute convertAttribute(IdentityObjectAttribute attribute)
   {
      if (attribute instanceof Attribute)
      {
         return (Attribute)attribute;
      }
      else
      {
         return new SimpleAttribute(attribute);
      }
   }

   protected IdentityObjectAttribute convertAttribute(Attribute attribute)
   {
      if (attribute instanceof IdentityObjectAttribute)
      {
         return (IdentityObjectAttribute)attribute;
      }
      else
      {
         return new SimpleAttribute(attribute);
      }
   }

   protected void checkNotNullArgument(Object arg, String name)
   {
      if (arg == null)
      {
         throw new IllegalArgumentException(name + " cannot be null");
      }
   }

   protected void checkObjectName(String name)
   {
      //TODO: extract this, let to define broader set of constraints and apply also in the
      //TODO: SPI to filter what comes fromdata stores
      if (name.contains("/"))
      {
         throw new IllegalArgumentException("name cannot contain '/' character");
      }
   }

}
