/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.identity.idm.cache;

import org.jboss.identity.idm.api.Attribute;
import org.jboss.identity.idm.api.Group;
import org.jboss.identity.idm.api.IdentitySearchCriteria;
import org.jboss.identity.idm.api.Role;
import org.jboss.identity.idm.api.User;
import org.jboss.identity.idm.api.query.GroupQuery;
import org.jboss.identity.idm.api.query.RoleQuery;
import org.jboss.identity.idm.api.query.UserQuery;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Cache provider for Identity API. Namespaces enable flexible use (per realm or per session)
 *
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public interface APICacheProvider
{

   /**
    * Generate realm namespace.
    *
    * @param realmId - if null will generate root namespace
    * @return
    */
   String getNamespace(String realmId);

   /**
    * Generate namespace based on realm and session ids.
    *
    * @param realmId if null will generate root namespace
    * @param sessionId if null will generate realm namespace
    * @return
    */
   String getNamespace(String realmId, String sessionId);


   // Persistence Manager

   
   void putUser(String ns, User user);

   User getUser(String ns, String id);

   void removeUser(String ns, String id);

   void putUsers(String ns, IdentitySearchCriteria criteria, Collection<User> users);

   Collection<User> getUsers(String ns, IdentitySearchCriteria criteria);

   void invalidateUsers(String ns);

   void putUserCount(String ns, int count);

   int getUserCount(String ns);

   void invalidateUserCount(String ns);

   //

   void putGroup(String ns, Group group);

   Group getGroup(String ns, String groupType, String groupName);

   void removeGroup(String ns, String groupType, String groupName);

   void putGroups(String ns, IdentitySearchCriteria criteria, Collection<Group> groups);

   void invalidateGroups(String ns);

   void putGroupCount(String ns, String groupType, int count);

   int getGroupCount(String ns, String groupType);

   void invalidateGroupCount(String ns, String groupType);


   // Attribute
   void putAttributes(String ns, String id, Map<String, Attribute> attributes);

   Map<String, Attribute> getAttributes(String ns, String id);


   // Queries

   void putUserQuery(String ns, UserQuery q, Collection<User> results);

   void putUserQueryList(String ns, UserQuery q, List<User> results);

   void putUserQueryUnique(String ns, UserQuery q, User user);

   Collection<User> getUserQuery(String ns, UserQuery q, Collection<User> results);

   List<User> getUserQueryList(String ns, UserQuery q, List<User> results);

   User getUserQueryUnique(String ns, UserQuery q, User user);

   void invalidateUserQueries(String ns);

   //
   void putGroupQuery(String ns, GroupQuery q, Collection<Group> results);

   void putGroupQueryList(String ns, GroupQuery q, List<Group> results);

   void putGroupQueryUnique(String ns, GroupQuery q, Group group);

   Collection<Group> getGroupQuery(String ns, GroupQuery q, Collection<Group> results);

   List<Group> getGroupQueryList(String ns, GroupQuery q, List<Group> results);

   Group getGroupQueryUnique(String ns, GroupQuery q, Group group);

   void invalidateGroupQueries(String ns);

   //
   void putRoleQuery(String ns, RoleQuery q, Collection<Role> results);

   void putRoleQueryList(String ns, RoleQuery q, List<Role> results);

   void putRoleQueryUnique(String ns, RoleQuery q, Role role);

   Collection<Role> getRoleQuery(String ns, RoleQuery q, Collection<Role> results);

   List<Role> getRoleQueryList(String ns, RoleQuery q, List<Role> results);

   Role getRoleQueryUnique(String ns, RoleQuery q, Role role);

   void invalidateRoleQueries(String ns);

}
