/**
 * 
 */
package org.jboss.identity.idm.integration.jboss5;

import java.net.URL;
import java.util.Collection;
import java.util.Iterator;
import java.util.logging.Logger;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.TransactionManager;

import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.AnnotationConfiguration;
import org.hibernate.cfg.Configuration;
import org.hibernate.exception.SQLGrammarException;
import org.hibernate.metadata.ClassMetadata;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.hibernate.tool.hbm2ddl.SchemaUpdate;
import org.jboss.identity.idm.api.cfg.IdentityConfiguration;
import org.jboss.identity.idm.common.transaction.TransactionManagerProvider;
import org.jboss.identity.idm.common.transaction.Transactions;
import org.jboss.identity.idm.integration.jboss5.jaxb2.HibernateInitializerType;
import org.jboss.portal.common.net.URLTools;
import org.jboss.portal.common.util.CLResourceLoader;
import org.jboss.portal.common.util.LoaderResource;

import bsh.EvalError;
import bsh.Interpreter;

/**
 * 
 * Using the Hibernate built-in SchemaExport.
 * 
 * @author  Jeff Yu
 * @author  Boleslaw Dawidowicz
 * @author  Julien Viet
 *
 */
public class HibernatePopulator {
	
	private static Logger logger = Logger.getLogger(HibernatePopulator.class.getName());
	
	/** doCheck result - schema ok */
	private static final int RESULT_NONE = 0;

	/** doCheck result - schema need updates */
	private static final int RESULT_UPDATE = 1;

	/** doCheck result - schema not exist */
    private static final int RESULT_CREATE = 2;

	private Configuration config;
	
	private SessionFactory sessionFactory;
	
	private HibernateInitializerType type;
	
	private String hibernateSessionFactoryJNDIName;
	
	private IdentityConfiguration identityConfiguration;
	
	
	public HibernatePopulator(HibernateInitializerType type, IdentityConfiguration identityConfig) {
		this.type = type;
		this.identityConfiguration = identityConfig;
	}
	
	
	public void populateSchema() throws Exception {
		
	      // If hibernate config is present than create SessionFactory manually and register it into identity config
	      if (type.getHibernateConfigLocation() != null)
	      {
	         URL configURL = Thread.currentThread().getContextClassLoader().getResource(type.getHibernateConfigLocation());
	         if (configURL == null)
	         {
	            throw new Exception("The config " + type.getHibernateConfigLocation() + " does not exist");
	         }

	         if (!URLTools.exists(configURL))
	         {
	            throw new Exception("The config " + configURL + " does not exist");
	         }

	         config = new AnnotationConfiguration().configure(configURL);
	         
	         createHibernateSessionFactory();
	         
	         Context context = new InitialContext();
	         if (getHibernateSessionFactoryJNDIName() != null)
	         {
	            try
	            {
	               context.bind(getHibernateSessionFactoryJNDIName(), sessionFactory);
	            }
	            catch (NamingException e)
	            {
	            	logger.severe(e.getMessage());
	               
	            }
	         }

	         if (type.getHibernateSessionFactoryRegistryName() != null)
	         {
	            identityConfiguration.getIdentityConfigurationRegistry().
	               register(sessionFactory, type.getHibernateSessionFactoryRegistryName());
	         }

	         if (type.isDoChecking())
	         {
	            //check the schema
	            int check = doCheck();
	            switch (check)
	            {
	               case RESULT_NONE:
	                  break;
	               case RESULT_UPDATE:
	                  updateSchema();
	                  break;
	               case RESULT_CREATE:
	                  createSchema();
	                  createContent();

	                  break;
	            }
	         } else {
	        	 createSchema();
	         }
	      }
		
	}
	
	
    private int doCheck()
    {
      Session session = null;
      int numOfChecks = 0;
      int bad = 0;
      try
      {
         session = sessionFactory.openSession();
         Collection<ClassMetadata> values = sessionFactory.getAllClassMetadata().values();
         numOfChecks = values.size();
         for (Iterator<ClassMetadata> i = values.iterator(); i.hasNext();)
         {
            ClassMetadata cmd = (ClassMetadata)i.next();
            Query query = session.createQuery("from " + cmd.getEntityName());
            query.setFirstResult(0);
            query.setMaxResults(0);
            try
            {
               query.list();
            }
            catch (SQLGrammarException e)
            {
               // We consider that exception means that the schema does not exist
               bad++;
            }
         }
      }
      finally
      {
         sessionFactory.close();
      }
      // There was no sql grammar exception - schema is ok!
      if (bad == 0)
      {
         logger.fine("The schema was checked as valid");
         //do nothing
         return RESULT_NONE;
      }
      // There is no existing valid schema;
      else if (bad == numOfChecks)
      {
         logger.fine("The schema was checked as not exists");
         // Totaly invalid schema
         return RESULT_CREATE;
      }
      // Schema is partialy corrupted
      else if (bad < numOfChecks)
      {
         // Schema needs updates;
         logger.fine("The schema was checked as need updates");
         return RESULT_UPDATE;
      }

      // If here something gone wrong...
      logger.fine("The schema was checked as need to be created");
      return RESULT_CREATE;
   }
	
    private void createSchema()
    {
      logger.fine("Creating database schema");
      SchemaExport export = new SchemaExport(config);
      export.create(false, true);
    }
   
    private void destroySchema()
    {
       logger.fine("Destroying database schema");
       SchemaExport export = new SchemaExport(config);
       export.drop(false, true);
    }
    
    private void updateSchema()
    {
       logger.fine("Updating database schema");
       SchemaUpdate update = new SchemaUpdate(config);
       update.execute(false, true);
    }
    
    public String getHibernateSessionFactoryJNDIName() {
		return hibernateSessionFactoryJNDIName;
	}


	public void setHibernateSessionFactoryJNDIName(
			String hibernateSessionFactoryJNDIName) {
		this.hibernateSessionFactoryJNDIName = hibernateSessionFactoryJNDIName;
	}


	private void createHibernateSessionFactory() throws Exception
    {
       // Set JNDI name if present and absent
       if (getHibernateSessionFactoryJNDIName() != null)
       {
          setPropertyIfAbsent("hibernate.session_factory_name", getHibernateSessionFactoryJNDIName());
       }

       sessionFactory = config.buildSessionFactory();
    }
	
	   private void setPropertyIfAbsent(String name, String value)
	   {
	      if (config.getProperty(name) == null)
	      {
	         config.setProperty(name, value);
	      }
	   }
	   
	   
	   private void createContent()
	   {
	      LoaderResource setupResource = null;

	      if (type.getSetupLocation() != null && setupResource == null)
	      {
	         setupResource = new CLResourceLoader().getResource(type.getSetupLocation());
	      }

	      if (setupResource != null)
	      {
	         if (setupResource.exists())
	         {
	            try
	            {
	               logger.info("Creating database content");
	               final String script = setupResource.asString("UTF-8");

	               // Create an interpreter and configures it
	               final Interpreter interpreter = new Interpreter();
	               interpreter.setClassLoader(Thread.currentThread().getContextClassLoader());
	               interpreter.setOut(System.out);
	               interpreter.set("SessionFactory", sessionFactory);

	               TransactionManager tm = TransactionManagerProvider.JBOSS_PROVIDER.getTransactionManager();
	               Transactions.required(tm, new Transactions.Runnable()
	               {
	                  public Object run() throws Exception
	                  {
	                     interpreter.eval(script);
	                     return null;
	                  }
	               });
	            }
	            catch (EvalError e)
	            {
	               logger.info("Error in the bsh script: " + e);
	            }
	            catch (IllegalStateException e)
	            {
	               logger.info("Cannot load setup script: " + e);
	            }
	            catch (Exception e)
	            {
	               logger.info("Error in bsh script execution: " + e);
	            }
	         }
	         else
	         {
	            logger.info("There is a setup URL but the not valid " + setupResource);
	         }
	      }
	   }

}
