/**
 * 
 */
package org.jboss.identity.idm.integration.jboss5;

import java.io.InputStream;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import javax.transaction.TransactionManager;

import org.hibernate.SessionFactory;
import org.hibernate.cfg.AnnotationConfiguration;
import org.hibernate.dialect.Dialect;
import org.hibernate.dialect.DialectFactory;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.vfs.spi.deployer.AbstractSimpleVFSRealDeployer;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.identity.idm.api.IdentitySessionFactory;
import org.jboss.identity.idm.api.cfg.IdentityConfiguration;
import org.jboss.identity.idm.common.exception.IdentityException;
import org.jboss.identity.idm.common.transaction.Transactions;
import org.jboss.identity.idm.impl.configuration.IdentityConfigurationImpl;
import org.jboss.identity.idm.impl.configuration.jaxb2.JAXB2IdentityConfiguration;
import org.jboss.identity.idm.integration.jboss5.jaxb2.HibernateDeployerType;
import org.jboss.identity.idm.integration.jboss5.jaxb2.HibernateInitializerType;
import org.jboss.identity.idm.integration.jboss5.jaxb2.JbossIDMDeployerType;
import org.jboss.identity.idm.integration.jboss5.jaxb2.SqlInitializerType;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityConfigurationMetaData;
import org.jboss.identity.idm.spi.configuration.metadata.IdentityStoreConfigurationMetaData;


/**
 * 
 * 
 * @author  Jeff Yu
 *
 */
public class IDMDeployer extends AbstractSimpleVFSRealDeployer<IDMMetadata> {

	private static final Logger logger = Logger.getLogger(IDMDeployer.class.getName());
	
	private static final String HIBERNATE_CONFIGFILE = "hibernateConfiguration";
	
	private IdentitySessionFactory idSF;
	
	private SessionFactory hibernateSF;
	
	private TransactionManager transactionManager;
	
	private IdentityConfiguration identityConfiguration;
	
	public IDMDeployer() {
		super(IDMMetadata.class);
	}
	
	@Override
	public void deploy(VFSDeploymentUnit deploymentUnit, IDMMetadata metadata) throws DeploymentException {
		JbossIDMDeployerType config = metadata.getDeploperType();		
		
		try {			
			InputStream is = deploymentUnit.getClassLoader().getResourceAsStream(config.getIdmConfigFile());			
			IdentityConfigurationMetaData identityMetadata = JAXB2IdentityConfiguration.createConfigurationMetaData(is);
			identityConfiguration = new IdentityConfigurationImpl().configure(identityMetadata);
			
			if (config.getHibernateDeployer() != null) {
				deployHibernateConfigurationFile(config, identityConfiguration);
			}
			
			if (config.getInitializers() != null) {
				initializeDB(config, identityMetadata, identityConfiguration);					
			}

		      try {
		          Transactions.required(transactionManager, new Transactions.Runnable()
		          {
		             public Object run() throws Exception
		             {
		                idSF = identityConfiguration.buildIdentitySessionFactory();
		                return null;
		             }
		          });
		       } catch (Exception e) {
		         throw new Exception("Cannot create IdentitySessionFactory", e); 
		       }
		       
		       InitialContext context = new InitialContext();
		       context.bind(config.getJNDIName(), idSF);
		 		logger.info("Started [" + metadata.getDeployerFileName() +  "]  IDM SessionFactory at JNDI [" + config.getJNDIName() + "]");
		
		} catch (Exception e) {
			throw new DeploymentException(e);
		}
		
	}

	private void initializeDB(JbossIDMDeployerType config, IdentityConfigurationMetaData identityMetadata,
			final IdentityConfiguration identityConfiguration) throws Exception {
		logger.fine("starting to populate the schema into db");
		
		String datasource = config.getInitializers().getDatasource();
		checkTargetDB(config.getInitializers().getDatasource());
		
		HibernateInitializerType hibernateInitializer = config.getInitializers().getHibernateInitializer();
		SqlInitializerType sqlInitializer = config.getInitializers().getSqlInitializer();
		
		if (hibernateInitializer != null) {
			for (IdentityStoreConfigurationMetaData store : identityMetadata.getIdentityStores()) {
				String hibernateConfigFile = store.getOptionSingleValue(HIBERNATE_CONFIGFILE);
				if (hibernateConfigFile != null && !"".equals(hibernateConfigFile.trim())) {
					logger.fine("starting to populate the schema from file [" + hibernateConfigFile + "]");
					HibernatePopulator hibernatePopulator = new HibernatePopulator(hibernateInitializer, identityConfiguration);
					hibernatePopulator.populateSchema();
				}
			}
			
		}else if (sqlInitializer != null) {
			logger.fine("starting to populate the schema from script file [" + sqlInitializer.getSqlFile() + "]");
			SQLPopulator sqlPopulator = new SQLPopulator(datasource, sqlInitializer.getSqlFile(), sqlInitializer.getExitSQL());
			sqlPopulator.populateSchema();
		}
	}

	private void deployHibernateConfigurationFile(JbossIDMDeployerType config,
			final IdentityConfiguration identityConfiguration) throws NamingException, IdentityException {
		HibernateDeployerType hibernateConfig = config.getHibernateDeployer();
		hibernateSF = new AnnotationConfiguration().
																		configure(hibernateConfig.getHibernateConfiguration()).buildSessionFactory();
		if (hibernateConfig.getHibernateSessionFactoryJNDIName() != null) {
			InitialContext context = new InitialContext();
			context.bind(hibernateConfig.getHibernateSessionFactoryJNDIName(), hibernateSF);
			logger.fine("Registered the Hibernate Session Factory in JNDI of " + hibernateConfig.getHibernateSessionFactoryJNDIName());
		}
		if (hibernateConfig.getHibernateSessionFactoryRegistryName() != null) {
			identityConfiguration.getIdentityConfigurationRegistry().register(hibernateSF, hibernateConfig.getHibernateSessionFactoryRegistryName());
			logger.fine("Registered the Hibernate Session Factory in Identity Registration of " + hibernateConfig.getHibernateSessionFactoryRegistryName());
		}
	}
	
	
	@Override
	public void undeploy(VFSDeploymentUnit deploymentUnit, IDMMetadata metadata) {
		if (idSF != null) {
			idSF.close();
		}
		
		try {
			InitialContext context = new InitialContext();
			context.unbind(metadata.getDeploperType().getJNDIName());
			
			if (metadata.getDeploperType().getHibernateDeployer() != null) {
				
				if (hibernateSF != null) {
					hibernateSF.close();
				}
				
				HibernateDeployerType hibernateDeployer = metadata.getDeploperType().getHibernateDeployer();
				if (hibernateDeployer.getHibernateSessionFactoryJNDIName() != null) {
					context.unbind(hibernateDeployer.getHibernateSessionFactoryJNDIName());
				}
				if (hibernateDeployer.getHibernateSessionFactoryRegistryName() != null) {
					identityConfiguration.getIdentityConfigurationRegistry().unregister(hibernateDeployer.getHibernateSessionFactoryRegistryName());
				}
			}
		} catch (Exception e) {
			logger.severe(e.getMessage());
			throw new RuntimeException(e);
		}
		
		logger.info("Stopped [" + metadata.getDeployerFileName() +  "] IDM SessionFactory at JNDI [" + metadata.getDeploperType().getJNDIName() + "]");

	}
	
	
	private void checkTargetDB(String datasource) {
		
        Connection conn = null;
        Dialect dialectName = null;
        try
        {
        	DataSource ds = (DataSource)new InitialContext().lookup(datasource);
           conn = ds.getConnection();
           DatabaseMetaData meta = conn.getMetaData();
           String databaseName = meta.getDatabaseProductName();
           int databaseMajorVersion = getDatabaseMajorVersion(meta);
           dialectName = DialectFactory.determineDialect(databaseName, databaseMajorVersion);
           logger.fine("Detected dialect " + dialectName + ", database is (" + databaseName + "," + databaseMajorVersion + ")");
        } catch (Exception e) {
        	logger.warning(e.getMessage());
        }
        finally
        {
           if (conn != null) {
        	   try {
				conn.close();
			} catch (SQLException e) {
				logger.log(Level.SEVERE, "error in closing the connection", e);
			}
           }
        }
	}
        
	private int getDatabaseMajorVersion(DatabaseMetaData meta)
	{
	   try
	   {
	      Method gdbmvMethod = DatabaseMetaData.class.getMethod("getDatabaseMajorVersion", (Class[])null);
	      return ((Integer)gdbmvMethod.invoke(meta, (Object[])null)).intValue();
	   }
	   catch (NoSuchMethodException nsme)
	   {
	      return 0;
	   }
	   catch (Throwable t)
	   {
	      logger.fine("could not get database version from JDBC metadata");
	      return 0;
	   }
	}

	public TransactionManager getTransactionManager() {
		return transactionManager;
	}

	public void setTransactionManager(TransactionManager transactionManager) {
		this.transactionManager = transactionManager;
	}

	
	

}
