/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.bindings.servlets;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import org.apache.log4j.Logger;
import org.jboss.identity.federation.api.saml.v2.common.IDGenerator;
import org.jboss.identity.federation.api.saml.v2.response.SAML2Response;
import org.jboss.identity.federation.core.saml.v2.factories.SAMLAssertionFactory;
import org.jboss.identity.federation.core.saml.v2.factories.SOAPFactory;
import org.jboss.identity.federation.core.saml.v2.holders.IssuerInfoHolder;
import org.jboss.identity.federation.core.saml.v2.util.JAXBElementMappingUtil;
import org.jboss.identity.federation.core.saml.v2.util.SOAPSAMLXACMLUtil;
import org.jboss.identity.federation.org.xmlsoap.schemas.soap.envelope.Body;
import org.jboss.identity.federation.org.xmlsoap.schemas.soap.envelope.Envelope;
import org.jboss.identity.federation.org.xmlsoap.schemas.soap.envelope.Fault;
import org.jboss.identity.federation.saml.v2.assertion.AssertionType;
import org.jboss.identity.federation.saml.v2.profiles.xacml.assertion.XACMLAuthzDecisionStatementType;
import org.jboss.identity.federation.saml.v2.profiles.xacml.protocol.XACMLAuthzDecisionQueryType;
import org.jboss.identity.federation.saml.v2.protocol.RequestAbstractType;
import org.jboss.security.xacml.core.JBossPDP;
import org.jboss.security.xacml.core.JBossRequestContext;
import org.jboss.security.xacml.core.model.context.RequestType;
import org.jboss.security.xacml.core.model.context.ResponseType;
import org.jboss.security.xacml.core.model.context.ResultType;
import org.jboss.security.xacml.interfaces.PolicyDecisionPoint;
import org.jboss.security.xacml.interfaces.RequestContext;
import org.jboss.security.xacml.interfaces.ResponseContext;

/**
 * Servlet that can read SOAP 1.1 messages that contain
 * an XACML query in saml payload
 * @author Anil.Saldhana@redhat.com
 * @since Jan 27, 2009
 */
public class SOAPSAMLXACMLServlet extends HttpServlet
{   
   private static Logger log = Logger.getLogger(SOAPSAMLXACMLServlet.class);
   
   private static final long serialVersionUID = 1L;
   
   String policyConfigFileName = null;
   
   String issuerId = null;
   String issuer = null;

   public void init() throws ServletException
   {    
      issuerId = getServletContext().getInitParameter("issuerID");
      if(issuerId == null)
         issuerId = "issue-id:1";
      
      issuer = getServletContext().getInitParameter("issuer");
      if(issuer == null)
         issuer = "urn:jboss-identity";
      
      policyConfigFileName = getServletContext().getInitParameter("policyConfigFileName");
      if(policyConfigFileName == null)
         policyConfigFileName = "policyConfig.xml"; 
      
      super.init();     
   }
   
   public void init(ServletConfig config)  throws ServletException
   {
      super.init(config); 
   }

   
   @SuppressWarnings("unchecked")
   @Override
   protected void service(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException
   {
      JAXBElement<RequestAbstractType> jaxbRequestType = null;
      
      Envelope envelope = null;
      XACMLAuthzDecisionQueryType xacmlRequest = null;
      
      try
      {
         Unmarshaller un = SOAPSAMLXACMLUtil.getUnmarshaller();
         Object unmarshalledObject = un.unmarshal(req.getInputStream());
         
         if(unmarshalledObject instanceof JAXBElement)
         {
            JAXBElement<?> jaxbElement = (JAXBElement<?>) unmarshalledObject;
            Object element = jaxbElement.getValue();
            if(element instanceof Envelope)
            {
               envelope = (Envelope)element; 
               Body soapBody = envelope.getBody(); 
               jaxbRequestType = (JAXBElement<RequestAbstractType>)soapBody.getAny().get(0);
               xacmlRequest = (XACMLAuthzDecisionQueryType) jaxbRequestType.getValue();
            }
            else if(element instanceof XACMLAuthzDecisionQueryType)
            {
               xacmlRequest = (XACMLAuthzDecisionQueryType) element;
            }
         }
         if(xacmlRequest == null)
            throw new IOException("XACML Request not parsed"); 

         RequestType requestType = xacmlRequest.getRequest();
         
         RequestContext requestContext = new JBossRequestContext();
         requestContext.setRequest(requestType);
         
         ResponseContext responseContext = getPDP().evaluate(requestContext);
         
         ResponseType responseType = new ResponseType();
         ResultType resultType = responseContext.getResult();
         responseType.getResult().add(resultType);

         XACMLAuthzDecisionStatementType xacmlStatement = SOAPSAMLXACMLUtil.createXACMLAuthzDecisionStatementType();
         xacmlStatement.setRequest(requestType);
         xacmlStatement.setResponse(responseType);
         
         //Place the xacml statement in an assertion
         //Then the assertion goes inside a SAML Response
         
         SAML2Response saml2Response = new SAML2Response();
         IssuerInfoHolder issuerInfo = new IssuerInfoHolder(this.issuer);
         
         AssertionType assertion = SAMLAssertionFactory.getObjectFactory().createAssertionType();
         assertion.getStatementOrAuthnStatementOrAuthzDecisionStatement().add(xacmlStatement);
         
         JAXBElement<?> jaxbResponse = JAXBElementMappingUtil.get(saml2Response.createResponseType(IDGenerator.create("ID_"), issuerInfo, assertion));
         
         //Create a SOAP Envelope to hold the SAML response
         envelope = this.createEnvelope(jaxbResponse); 
      }
      catch (JAXBException e)
      {
         log.error("Exception parsing SOAP:", e);  
         envelope = this.createEnvelope(this.createFault("Parsing Error:"+e.getMessage()));
      }
      catch (PrivilegedActionException e)
      {
         log.error("Exception getting PDP:", e); 
         envelope = this.createEnvelope(this.createFault("PDP Setup Error:"+e.getMessage()));
      } 
      catch (Exception e)
      { 
         log.error("Exception:", e); 
         envelope = this.createEnvelope(this.createFault("Server Error:"+e.getMessage()));
      } 
      finally
      {
         resp.setContentType("text/xml;charset=utf-8");;
         OutputStream os = resp.getOutputStream(); 
         try
         {
            if(envelope == null)
               throw new IllegalStateException("SOAPEnvelope is null");
            JAXBElement<?> jaxbEnvelope = JAXBElementMappingUtil.get(envelope);
            Marshaller marshaller = SOAPSAMLXACMLUtil.getMarshaller();
            marshaller.marshal(jaxbEnvelope, os);  
         }
         catch (JAXBException e)
         {
            log("marshalling exception",e);
         }  
      } 
   } 
   
   private PolicyDecisionPoint getPDP() throws PrivilegedActionException
   {
      ClassLoader tcl = AccessController.doPrivileged(new PrivilegedExceptionAction<ClassLoader>()
      {
         public ClassLoader run() throws Exception
         {
             return Thread.currentThread().getContextClassLoader();
         }
      });
      InputStream is = tcl.getResourceAsStream(this.policyConfigFileName);
      if(is == null)
         throw new IllegalStateException(policyConfigFileName  + " could not be located");
      return new JBossPDP(is); 
   } 
   
   private Envelope createEnvelope(Object obj)
   {
      Envelope envelope = SOAPFactory.getObjectFactory().createEnvelope();
      Body body = SOAPFactory.getObjectFactory().createBody();
      body.getAny().add(obj); 
      envelope.setBody(body);
      return envelope;
   }
   
   private JAXBElement<Fault> createFault(String msg)
   {
      Fault fault = SOAPFactory.getObjectFactory().createFault();
      fault.setFaultstring(msg);
      return SOAPFactory.getObjectFactory().createFault(fault); 
   }
}