/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.core.saml.v2.factories;

import java.net.URL;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.jboss.identity.federation.core.saml.v2.constants.JBossSAMLURIConstants;
import org.jboss.identity.federation.saml.v2.assertion.AssertionType;
import org.jboss.identity.federation.saml.v2.assertion.AttributeStatementType;
import org.jboss.identity.federation.saml.v2.assertion.AttributeType;
import org.jboss.identity.federation.saml.v2.assertion.NameIDType;
import org.jboss.identity.federation.saml.v2.assertion.ObjectFactory;
import org.jboss.identity.federation.saml.v2.assertion.SubjectConfirmationDataType;
import org.jboss.identity.federation.saml.v2.assertion.SubjectConfirmationType;
import org.jboss.identity.federation.saml.v2.assertion.SubjectType;

/**
 * Base methods for the factories
 * @author Anil.Saldhana@redhat.com
 * @since Dec 9, 2008
 */
public class JBossSAMLBaseFactory
{
   private static ObjectFactory assertionObjectFactory = new ObjectFactory();
   
   public static AssertionType createAssertion()
   {
      return assertionObjectFactory.createAssertionType();  
   }
   
   public static AttributeStatementType createAttributeStatement()
   {
      return assertionObjectFactory.createAttributeStatementType();
   }
   
   /**
    * Create an attribute type given a role name
    * @param roleName
    * @return
    */
   public static AttributeType createAttributeForRole(String roleName)
   {
      AttributeType att = assertionObjectFactory.createAttributeType();
      att.setFriendlyName("role");
      att.setName("role");
      att.setNameFormat(JBossSAMLURIConstants.ATTRIBUTE_FORMAT_BASIC.get());
      
      //rolename 
      att.getAttributeValue().add(roleName);
      
      return att;
   }
   
   /**
    * Create an AttributeStatement given an attribute
    * @param attributeValue
    * @return
    */
   public static AttributeStatementType createAttributeStatement(String attributeValue)
   {
      AttributeStatementType attribStatement = assertionObjectFactory.createAttributeStatementType();
      AttributeType att = assertionObjectFactory.createAttributeType();
      JAXBElement<Object> attValue = assertionObjectFactory.createAttributeValue(attributeValue);
      att.getAttributeValue().add(attValue);
      attribStatement.getAttributeOrEncryptedAttribute().add(att);
      return attribStatement;
   }
   
   public static NameIDType createNameID()
   {
      return assertionObjectFactory.createNameIDType();
   }
   
   public static JAXBElement<NameIDType> createNameID(NameIDType nameIDType)
   {
      return assertionObjectFactory.createNameID(nameIDType);
   }
   
   public static SubjectType createSubject()
   {
      SubjectType subjectType = assertionObjectFactory.createSubjectType();
      return subjectType;
   }
   
   public static SubjectConfirmationType createSubjectConfirmation(String method)
   {
      SubjectConfirmationType sct = assertionObjectFactory.createSubjectConfirmationType();
      sct.setMethod(method);
      return sct;
   }
   
   public static JAXBElement<SubjectConfirmationType> createSubjectConfirmation(SubjectConfirmationType sct)
   {
      return assertionObjectFactory.createSubjectConfirmation(sct);
   }
   
   public static SubjectConfirmationDataType createSubjectConfirmationData(String inResponseTo, 
         String destinationURI, XMLGregorianCalendar issueInstant)
   {
      SubjectConfirmationDataType subjectConfirmationData = assertionObjectFactory.createSubjectConfirmationDataType();
      subjectConfirmationData.setInResponseTo(inResponseTo);
      subjectConfirmationData.setRecipient(destinationURI);
      subjectConfirmationData.setNotBefore(issueInstant);
      subjectConfirmationData.setNotOnOrAfter(issueInstant);
      
      return subjectConfirmationData;
   }
   
   /**
    * Get a UUID String
    * @return
    */
   public static String createUUID()
   {
      return java.util.UUID.randomUUID().toString(); 
   }
   
   public static ObjectFactory getObjectFactory()
   {
      return assertionObjectFactory;
   }
   
   /**
    * Return the NameIDType for the issuer
    * @param issuerID
    * @return
    */
   public static NameIDType getIssuer(String issuerID)
   {
      NameIDType nid = assertionObjectFactory.createNameIDType();
      nid.setValue(issuerID);
      return nid;
   }
   
   /**
    * Get the JAXB Marshaller
    * @param pkgName The package name for the jaxb context  
    * @return Marshaller
    * @throws Exception
    */
   public static Marshaller getMarshaller(String pkgName)
   throws Exception
   {
      if(pkgName == null)
         throw new IllegalArgumentException("pkgName is null");
      
      JAXBContext jc = JAXBContext.newInstance(pkgName);
      Marshaller marshaller = jc.createMarshaller();
      marshaller.setProperty(Marshaller.JAXB_ENCODING, "UTF-8");
      return marshaller;
   }
  
   /**
    * Get the JAXB Marshaller
    * @param pkgName The package name for the jaxb context
    * @param schemaLocation location of the schema to validate against 
    * @return Marshaller
    * @throws Exception
    */
   public static Marshaller getValidatingMarshaller(String pkgName, String schemaLocation)
   throws Exception
   {
      Marshaller marshaller = getMarshaller(pkgName);
      marshaller.setProperty(Marshaller.JAXB_ENCODING, "UTF-8");
      //Validate against schema
      ClassLoader tcl = SecurityActions.getContextClassLoader();
      URL schemaURL = tcl.getResource(schemaLocation);
      if(schemaURL == null)
         throw new IllegalStateException("Schema URL is null:" + schemaLocation);
      SchemaFactory scFact = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
      Schema schema = scFact.newSchema(schemaURL);
      marshaller.setSchema(schema); 
 
      return marshaller;
   }
   
   /**
    * Get the JAXB Unmarshaller
    * @param pkgName The package name for the jaxb context 
    * @return unmarshaller
    * @throws Exception
    */
   public static Unmarshaller getUnmarshaller(String pkgName) throws Exception
   {
      if(pkgName == null)
         throw new IllegalArgumentException("pkgName is null");
      JAXBContext jc = JAXBContext.newInstance(pkgName);
      return jc.createUnmarshaller();
   }
   
   /**
    * Get the JAXB Unmarshaller
    * @param pkgName The package name for the jaxb context
    * @param schemaLocation location of the schema to validate against 
    * @return unmarshaller
    * @throws Exception
    */
   public static Unmarshaller getValidatingUnmarshaller(String pkgName, String schemaLocation)
   throws Exception
   { 
      Unmarshaller unmarshaller = getUnmarshaller(pkgName); 
      //Validate against schema
      ClassLoader tcl = SecurityActions.getContextClassLoader();
      URL schemaURL = tcl.getResource(schemaLocation);
      if(schemaURL == null)
         throw new IllegalStateException("Schema URL is null:" + schemaLocation);
      SchemaFactory scFact = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
      Schema schema = scFact.newSchema(schemaURL);
      unmarshaller.setSchema(schema); 
 
      return unmarshaller;
   }
}