/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.api.wstrust;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.util.JAXBSource;
import javax.xml.transform.Source;

import org.jboss.identity.federation.core.wstrust.BaseRequestSecurityToken;
import org.jboss.identity.federation.core.wstrust.BaseRequestSecurityTokenResponse;
import org.jboss.identity.federation.core.wstrust.RequestSecurityToken;
import org.jboss.identity.federation.core.wstrust.RequestSecurityTokenCollection;
import org.jboss.identity.federation.core.wstrust.RequestSecurityTokenResponse;
import org.jboss.identity.federation.core.wstrust.RequestSecurityTokenResponseCollection;
import org.jboss.identity.federation.ws.trust.ObjectFactory;
import org.jboss.identity.federation.ws.trust.RequestSecurityTokenCollectionType;
import org.jboss.identity.federation.ws.trust.RequestSecurityTokenResponseCollectionType;
import org.jboss.identity.federation.ws.trust.RequestSecurityTokenResponseType;
import org.jboss.identity.federation.ws.trust.RequestSecurityTokenType;

/**
 * <p>
 * This factory implements utility methods for converting between JAXB model objects and XML source.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class WSTrustJAXBFactory
{
   private static final WSTrustJAXBFactory instance = new WSTrustJAXBFactory();

   private final JAXBContext context;

   private final ObjectFactory factory;

   /**
    * <p>
    * Creates the {@code WSTrustJAXBFactory} singleton instance.
    * </p>
    */
   private WSTrustJAXBFactory()
   {
      StringBuffer packages = new StringBuffer();
      packages.append("org.jboss.identity.federation.ws.addressing");
      packages.append(":org.jboss.identity.federation.ws.policy");
      packages.append(":org.jboss.identity.federation.ws.trust");
      packages.append(":org.jboss.identity.federation.ws.wss.secext");
      packages.append(":org.jboss.identity.federation.ws.wss.utility");
      try
      {
         this.context = JAXBContext.newInstance(packages.toString());
         this.factory = new ObjectFactory();
      }
      catch (JAXBException e)
      {
         throw new RuntimeException(e.getMessage(), e);
      }
   }

   /**
    * <p>
    * Gets a reference to the singleton instance.
    * </p>
    * 
    * @return a reference to the {@code WSTrustJAXBFactory} instance.
    */
   public static WSTrustJAXBFactory getInstance()
   {
      return instance;
   }

   /**
    * <p>
    * Creates a {@code BaseRequestSecurityToken} from the specified XML source.
    * </p>
    * 
    * @param request the XML source containing the security token request message.
    * @return the constructed {@code BaseRequestSecurityToken} instance. It will be an instance of
    *         {@code RequestSecurityToken} the message contains a single token request, and an instance of
    *         {@code RequestSecurityTokenCollection} if multiples requests are being made in the same message.
    */
   public BaseRequestSecurityToken parseRequestSecurityToken(Source request)
   {
      try
      {
         Unmarshaller unmarshaller = this.context.createUnmarshaller();
         Object object = unmarshaller.unmarshal(request);

         // check the type of the unmarshalled object.
         if (object instanceof RequestSecurityTokenType)
            return new RequestSecurityToken((RequestSecurityTokenType) object);
         else if (object instanceof RequestSecurityTokenCollectionType)
            return new RequestSecurityTokenCollection((RequestSecurityTokenCollectionType) object);
         else if (object instanceof JAXBElement)
         {
            JAXBElement<?> element = (JAXBElement<?>) unmarshaller.unmarshal(request);
            // is this a single token request or a collection of token requests
            if (element.getDeclaredType().equals(RequestSecurityTokenType.class))
               return new RequestSecurityToken((RequestSecurityTokenType) element.getValue());
            else if (element.getDeclaredType().equals(RequestSecurityTokenCollectionType.class))
               return new RequestSecurityTokenCollection((RequestSecurityTokenCollectionType) element.getValue());
            else
               throw new RuntimeException("Invalid request type: " + element.getDeclaredType());
         }
         else
            throw new RuntimeException("Invalid request type: " + object.getClass().getName());
      }
      catch (Exception e)
      {
         throw new RuntimeException("Failed to unmarshall security token request", e);
      }
   }

   /**
    * <p>
    * Creates a {@code BaseRequestSecurityTokenResponse} from the specified XML source.
    * </p>
    * 
    * @param response the XML source containing the security token response message.
    * @return the constructed {@code BaseRequestSecurityTokenResponse} instance. It should return an instance of
    *         {@code RequestSecurityTokenResponseCollection} according to the specification, but we allow a single
    *         response to be returned in the form of a {@code RequestSecurityTokenResponse} instance.
    */
   public BaseRequestSecurityTokenResponse parseRequestSecurityTokenResponse(Source response)
   {
      try
      {
         Unmarshaller unmarshaller = this.context.createUnmarshaller();
         Object object = unmarshaller.unmarshal(response);
         // check the type of the response object.
         if (object instanceof RequestSecurityTokenResponseType)
            return new RequestSecurityTokenResponse((RequestSecurityTokenResponseType) object);
         else if (object instanceof RequestSecurityTokenResponseCollectionType)
            return new RequestSecurityTokenResponseCollection((RequestSecurityTokenResponseCollectionType) object);
         else if (object instanceof JAXBElement)
         {
            JAXBElement<?> element = (JAXBElement<?>) unmarshaller.unmarshal(response);
            // is this a single token response or a collection of token responses
            if (element.getDeclaredType().equals(RequestSecurityTokenResponseType.class))
               return new RequestSecurityTokenResponse((RequestSecurityTokenResponseType) element.getValue());
            else if (element.getDeclaredType().equals(RequestSecurityTokenResponseCollectionType.class))
               return new RequestSecurityTokenResponseCollection((RequestSecurityTokenResponseCollectionType) element
                     .getValue());
            else
               throw new RuntimeException("Invalid response type: " + element.getDeclaredType());
         }
         else
            throw new RuntimeException("Invalid response type: " + object.getClass().getName());
      }
      catch (Exception e)
      {
         throw new RuntimeException("Failed to unmarshall security token response", e);
      }
   }

   /**
    * <p>
    * Creates a {@code javax.xml.transform.Source} from the specified request object.
    * </p>
    * 
    * @param request a {@code BaseRequestSecurityToken} representing the object model of the security token request.
    * @return the constructed {@code Source} instance.
    */
   public Source marshallRequestSecurityToken(BaseRequestSecurityToken request)
   {
      JAXBElement<?> element = null;
      if (request instanceof RequestSecurityToken)
      {
         RequestSecurityToken requestSecurityToken = (RequestSecurityToken) request;
         element = this.factory.createRequestSecurityToken(requestSecurityToken.getDelegate());
      }
      else if (request instanceof RequestSecurityTokenCollection)
      {
         RequestSecurityTokenCollection collection = (RequestSecurityTokenCollection) request;
         element = this.factory.createRequestSecurityTokenCollection(collection.getDelegate());
      }
      else
         throw new RuntimeException("Failed to determine the type of the security token request");

      try
      {
         Marshaller marshaller = this.context.createMarshaller();
         return new JAXBSource(marshaller, element);
      }
      catch (JAXBException je)
      {
         throw new RuntimeException("Failed to marshall security token request", je);
      }
   }

   /**
    * <p>
    * Creates a {@code javax.xml.transform.Source} from the specified response object.
    * </p>
    * 
    * @param response a {@code BaseRequestSecurityTokenResponse} representing the object model of the security token
    *            response.
    * @return the constructed {@code Source} instance.
    */
   public Source marshallRequestSecurityTokenResponse(BaseRequestSecurityTokenResponse response)
   {
      JAXBElement<?> element = null;
      if (response instanceof RequestSecurityTokenResponse)
      {
         RequestSecurityTokenResponse requestSecurityTokenResponse = (RequestSecurityTokenResponse) response;
         element = this.factory.createRequestSecurityTokenResponse(requestSecurityTokenResponse.getDelegate());
      }
      else if (response instanceof RequestSecurityTokenResponseCollection)
      {
         RequestSecurityTokenResponseCollection collection = (RequestSecurityTokenResponseCollection) response;
         element = this.factory.createRequestSecurityTokenResponseCollection(collection.getDelegate());
      }
      else
         throw new RuntimeException("Failed to determine the type of the security token response");

      try
      {
         Marshaller marshaller = this.context.createMarshaller();
         return new JAXBSource(marshaller, element);
      }
      catch (JAXBException je)
      {
         throw new RuntimeException("Failed to marshall security token request", je);
      }
   }

}
