/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.api.wstrust;

import java.util.GregorianCalendar;
import java.util.Map;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.jboss.identity.federation.core.wstrust.Lifetime;
import org.jboss.identity.federation.ws.addressing.AttributedURIType;
import org.jboss.identity.federation.ws.addressing.EndpointReferenceType;
import org.jboss.identity.federation.ws.addressing.ObjectFactory;
import org.jboss.identity.federation.ws.policy.AppliesTo;
import org.jboss.identity.federation.ws.trust.RequestedReferenceType;
import org.jboss.identity.federation.ws.wss.secext.KeyIdentifierType;
import org.jboss.identity.federation.ws.wss.secext.SecurityTokenReferenceType;

/**
 * <p>
 * Utility class that provides methods for parsing/creating WS-Trust elements.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class WSTrustUtil
{

   /**
    * <p>
    * Creates an instance of {@code KeyIdentifierType} with the specified values.
    * </p>
    * 
    * @param valueType a {@code String} representing the identifier value type.
    * @param value a {@code String} representing the identifier value.
    * @return the constructed {@code KeyIdentifierType} instance.
    */
   public static KeyIdentifierType createKeyIdentifier(String valueType, String value)
   {
      KeyIdentifierType keyIdentifier = new KeyIdentifierType();
      keyIdentifier.setValueType(valueType);
      keyIdentifier.setValue(value);
      return keyIdentifier;
   }
   
   /**
    * <p>
    * Creates an instance of {@code RequestedReferenceType} with the specified values. This method first creates a
    * {@code SecurityTokenReferenceType} with the specified key identifier and attributes and then use this reference
    * to construct the {@code RequestedReferenceType} that is returned.
    * </p>
    * 
    * @param keyIdentifier the key identifier of the security token reference.
    * @param attributes the attributes to be set on the security token reference.
    * @return the constructed {@code RequestedReferenceType} instance.
    */
   public static RequestedReferenceType createRequestedReference(KeyIdentifierType keyIdentifier,
         Map<QName, String> attributes)
   {
      SecurityTokenReferenceType securityTokenReference = new SecurityTokenReferenceType();
      securityTokenReference.getAny().add(
            new org.jboss.identity.federation.ws.wss.secext.ObjectFactory().createKeyIdentifier(keyIdentifier));
      securityTokenReference.getOtherAttributes().putAll(attributes);
      RequestedReferenceType reference = new RequestedReferenceType();
      reference.setSecurityTokenReference(securityTokenReference);

      return reference;
   }
   
   /**
    * <p>
    * Creates an instance of {@code AppliesTo} using the specified endpoint address.
    * </p>
    * 
    * @param endpointURI a {@code String} representing the endpoint URI.
    * @return the constructed {@code AppliesTo} instance.
    */
   public static AppliesTo createAppliesTo(String endpointURI)
   {
      AttributedURIType attributedURI = new AttributedURIType();
      attributedURI.setValue(endpointURI);
      EndpointReferenceType reference = new EndpointReferenceType();
      reference.setAddress(attributedURI);
      AppliesTo appliesTo = new AppliesTo();
      appliesTo.getAny().add(new ObjectFactory().createEndpointReference(reference));

      return appliesTo;
   }

   /**
    * <p>
    * Parses the contents of the {@code AppliesTo} element and returns the address the uniquely identify the service
    * provider.
    * </p>
    * 
    * @param appliesTo the {@code AppliesTo} instance to be parsed.
    * @return the address of the service provider.
    */
   public static String parseAppliesTo(AppliesTo appliesTo)
   {
      EndpointReferenceType reference = null;
      for (Object obj : appliesTo.getAny())
      {
         if (obj instanceof EndpointReferenceType)
            reference = (EndpointReferenceType) obj;
         else if (obj instanceof JAXBElement)
         {
            JAXBElement<?> element = (JAXBElement<?>) obj;
            if (element.getName().getLocalPart().equalsIgnoreCase("EndpointReference"))
               reference = (EndpointReferenceType) element.getValue();
         }

         if (reference != null && reference.getAddress() != null)
            return reference.getAddress().getValue();
      }
      return null;
   }

   /**
    * <p>
    * Creates a {@code Lifetime} instance that specifies a range of time that starts at the current GMT time and has
    * the specified duration in milliseconds.
    * </p>
    * 
    * @param tokenTimeout the token timeout value (in milliseconds).
    * @return the constructed {@code Lifetime} instance.
    */
   public static Lifetime createDefaultLifetime(long tokenTimeout)
   {
      GregorianCalendar created = new GregorianCalendar();
      GregorianCalendar expires = new GregorianCalendar();
      expires.setTimeInMillis(created.getTimeInMillis() + tokenTimeout);

      return new Lifetime(created, expires);
   }

}
