/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.web.servlets;

import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.Principal;
import java.security.PublicKey;
import java.util.List;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;
import org.jboss.identity.federation.core.exceptions.ConfigurationException;
import org.jboss.identity.federation.core.exceptions.ParsingException;
import org.jboss.identity.federation.core.saml.v2.constants.JBossSAMLURIConstants;
import org.jboss.identity.federation.core.saml.v2.exceptions.IssueInstantMissingException;
import org.jboss.identity.federation.core.saml.v2.exceptions.IssuerNotTrustedException;
import org.jboss.identity.federation.saml.v2.protocol.AuthnRequestType;
import org.jboss.identity.federation.saml.v2.protocol.RequestAbstractType;
import org.jboss.identity.federation.web.config.IDPType;
import org.jboss.identity.federation.web.config.KeyProviderType;
import org.jboss.identity.federation.web.interfaces.RoleGenerator;
import org.jboss.identity.federation.web.interfaces.TrustKeyConfigurationException;
import org.jboss.identity.federation.web.interfaces.TrustKeyManager;
import org.jboss.identity.federation.web.interfaces.TrustKeyProcessingException;
import org.jboss.identity.federation.web.roles.DefaultRoleGenerator;
import org.jboss.identity.federation.web.util.ConfigurationUtil;
import org.jboss.identity.federation.web.util.IDPWebRequestUtil;
import org.jboss.identity.federation.web.util.RedirectBindingSignatureUtil;
import org.w3c.dom.Document;

/**
 * SAML Web Browser SSO - POST binding
 * @author Anil.Saldhana@redhat.com
 * @since Aug 13, 2009
 */
public class IDPServlet extends HttpServlet
{
   private static final long serialVersionUID = 1L;
   private static Logger log = Logger.getLogger(IDPServlet.class);
   private boolean trace = log.isTraceEnabled();

   public static final String PRINCIPAL_ID = "jboss_identity.principal"; 
   public static final String ROLES_ID = "jboss_identity.roles";

   protected IDPType idpConfiguration = null;

   private RoleGenerator rg = new DefaultRoleGenerator();

   private long assertionValidity = 5000; // 5 seconds in miliseconds

   private String identityURL = null;

   private TrustKeyManager keyManager;

   private Boolean ignoreIncomingSignatures = true;

   private Boolean signOutgoingMessages = true; 
   
   private ServletContext context = null;

   public Boolean getIgnoreIncomingSignatures()
   {
      return ignoreIncomingSignatures;
   }
   
   @Override
   public void init(ServletConfig config) throws ServletException
   {
      super.init(config);
      String configFile = "/WEB-INF/jboss-idfed.xml"; 
      context = config.getServletContext();
      InputStream is = context.getResourceAsStream(configFile);
      if(is == null)
         throw new RuntimeException(configFile + " missing");
      try
      {
         idpConfiguration = ConfigurationUtil.getIDPConfiguration(is);
         this.identityURL = idpConfiguration.getIdentityURL(); 
         log.trace("Identity Provider URL=" + this.identityURL); 
         this.assertionValidity = idpConfiguration.getAssertionValidity();
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
      if(this.signOutgoingMessages)
      {
         KeyProviderType keyProvider = this.idpConfiguration.getKeyProvider();
         if(keyProvider == null)
            throw new RuntimeException("Key Provider is null for context=" + context.getContextPath());
         
         try
         {
            ClassLoader tcl = SecurityActions.getContextClassLoader();
            String keyManagerClassName = keyProvider.getClassName();
            if(keyManagerClassName == null)
               throw new RuntimeException("KeyManager class name is null");
            
            Class<?> clazz = tcl.loadClass(keyManagerClassName);
            this.keyManager = (TrustKeyManager) clazz.newInstance();
            keyManager.setAuthProperties(keyProvider.getAuth());
            keyManager.setValidatingAlias(keyProvider.getValidatingAlias());
         }
         catch(Exception e)
         {
            log.error("Exception reading configuration:",e);
            throw new RuntimeException(e.getLocalizedMessage());
         }
         if(trace) 
            log.trace("Key Provider=" + keyProvider.getClassName()); 
      }
      
      //handle the role generator
      String rgString = config.getInitParameter("ROLE_GENERATOR");
      if(rgString != null && !"".equals(rgString))
         this.setRoleGenerator(rgString);
   }   

   @SuppressWarnings("unchecked")
   @Override
   protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
   {
      //Some issue with filters and servlets
      HttpSession session = request.getSession(false); 
      
      String samlMessage = (String) session.getAttribute("SAMLRequest");
      String relayState = (String) session.getAttribute("RelayState");
      
      String referer = request.getHeader("Referer"); 

      //See if the user has already been authenticated
      Principal userPrincipal = (Principal) session.getAttribute(PRINCIPAL_ID);

      if(userPrincipal == null)
      {
         //The sys admin has not set up the login servlet filters for the IDP
         if(trace) 
            log.trace("Login Filters have not been configured");
         response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR); 
      } 
      
      
      IDPWebRequestUtil webRequestUtil = new IDPWebRequestUtil(request, 
            idpConfiguration, keyManager);

      if(userPrincipal != null)
      {
         if(trace)
         {
            log.trace("Retrieved saml message and relay state from session");
            log.trace("saml message=" + samlMessage + "::relay state="+ relayState);
         }
         session.removeAttribute("SAMLRequest");

         if(relayState != null && relayState.length() > 0)
            session.removeAttribute("RelayState");

         //Send valid saml response after processing the request
         if(samlMessage != null)
         {
            //Get the SAML Request Message
            RequestAbstractType requestAbstractType =  null; 
            Document samlResponse = null;
            String destination = null;
            try
            {
               requestAbstractType = webRequestUtil.getSAMLRequest(samlMessage);
               boolean isPost = webRequestUtil.hasSAMLRequestInPostProfile();
               boolean isValid = validate(request.getRemoteAddr(),
                     request.getQueryString(),
                     new SessionHolder(samlMessage, null, null), isPost);
               
               if(!isValid)
                  throw new GeneralSecurityException("Validation check failed");

               webRequestUtil.isTrusted(requestAbstractType.getIssuer().getValue());

               
               List<String> roles = (List<String>) session.getAttribute(ROLES_ID); 
               if(roles == null) 
               {
                  roles = rg.generateRoles(userPrincipal);
                  session.setAttribute(ROLES_ID, roles);
               }
                  

               if(trace) 
                  log.trace("Roles have been determined:Creating response");

               AuthnRequestType art = (AuthnRequestType) requestAbstractType;
               destination = art.getAssertionConsumerServiceURL();

               samlResponse = 
                  webRequestUtil.getResponse(destination, 
                        userPrincipal, roles,
                        this.identityURL, this.assertionValidity, this.signOutgoingMessages); 
            }
            catch (IssuerNotTrustedException e)
            {
               if(trace) log.trace(e);

               samlResponse =
                  webRequestUtil.getErrorResponse(referer, 
                        JBossSAMLURIConstants.STATUS_REQUEST_DENIED.get(), 
                        this.identityURL, this.signOutgoingMessages);  
            }
            catch (ParsingException e)
            {
               if(trace) log.trace(e);

               samlResponse =
                  webRequestUtil.getErrorResponse(referer, 
                        JBossSAMLURIConstants.STATUS_AUTHNFAILED.get(), 
                        this.identityURL, this.signOutgoingMessages); 
            }
            catch (ConfigurationException e)
            {
               if(trace) log.trace(e); 

               samlResponse =
                  webRequestUtil.getErrorResponse(referer, 
                        JBossSAMLURIConstants.STATUS_AUTHNFAILED.get(), 
                        this.identityURL, this.signOutgoingMessages);
            }
            catch (IssueInstantMissingException e)
            {
               if(trace) log.trace(e); 

               samlResponse =
                  webRequestUtil.getErrorResponse(referer, 
                        JBossSAMLURIConstants.STATUS_AUTHNFAILED.get(), 
                        this.identityURL, this.signOutgoingMessages); 
            } 
            catch(GeneralSecurityException e)
            {
               if(trace) log.trace(e); 

               samlResponse =
                  webRequestUtil.getErrorResponse(referer, 
                        JBossSAMLURIConstants.STATUS_AUTHNFAILED.get(), 
                        this.identityURL, this.signOutgoingMessages); 
            }
            finally
            {
               try
               { 
                  if(this.signOutgoingMessages)
                     webRequestUtil.send(samlResponse, destination,relayState, response, true, 
                           this.keyManager.getSigningKey());
                  else
                     webRequestUtil.send(samlResponse, destination, relayState, response, false,null);
               }
               catch (ParsingException e)
               {
                  if(trace) log.trace(e);
               } 
               catch (GeneralSecurityException e)
               {
                  if(trace) log.trace(e);
               }
            } 
            return;
         }
         else
         {
            log.error("No SAML Request Message");
            if(trace) log.trace("Referer="+referer);

            try
            {
               sendErrorResponseToSP(referer, response, relayState, webRequestUtil);
            }
            catch (ConfigurationException e)
            {
               if(trace) log.trace(e);
            } 
         } 
      }
   }

   protected void sendErrorResponseToSP(String referrer, HttpServletResponse response, String relayState,
         IDPWebRequestUtil webRequestUtil) throws ServletException, IOException, ConfigurationException
   {
      if(trace) log.trace("About to send error response to SP:" + referrer);

      Document samlResponse =   
         webRequestUtil.getErrorResponse(referrer, JBossSAMLURIConstants.STATUS_RESPONDER.get(),
               this.identityURL, this.signOutgoingMessages);
      try
      {   
         if(this.signOutgoingMessages)
            webRequestUtil.send(samlResponse, referrer, relayState, response, true, 
                  this.keyManager.getSigningKey());
         else
            webRequestUtil.send(samlResponse, referrer, relayState, response, false,null);
      }
      catch (ParsingException e1)
      {
         throw new ServletException(e1);
      }
      catch (GeneralSecurityException e)
      { 
         throw new ServletException(e);
      } 
   } 

   @Override
   protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException
   {
      resp.sendError(HttpServletResponse.SC_METHOD_NOT_ALLOWED);
   }  
   

   protected class SessionHolder
   {
      String samlRequest;
      String signature;
      String sigAlg; 
      
      public SessionHolder(String req, String sig, String alg)
      {
         this.samlRequest = req;
         this.signature = sig;
         this.sigAlg = alg; 
      }
   }
   
   protected boolean validate(String remoteAddress, 
         String queryString,
         SessionHolder holder, boolean isPost) throws IOException, GeneralSecurityException
   {   
      if (holder.samlRequest == null || holder.samlRequest.length() == 0)
      {
         return false;
      }

      if (!this.ignoreIncomingSignatures && !isPost)
      {
         String sig = holder.signature;
         if (sig == null || sig.length() == 0)
         {
            log.error("Signature received from SP is null:" + remoteAddress);
            return false;
         }
         
         //Check if there is a signature   
         byte[] sigValue = RedirectBindingSignatureUtil.getSignatureValueFromSignedURL(queryString);
         if(sigValue == null)
            return false;
         
         PublicKey validatingKey;
         try
         {
            validatingKey = keyManager.getValidatingKey(remoteAddress);
         }
         catch (TrustKeyConfigurationException e)
         {
            throw new GeneralSecurityException(e.getCause());
         }
         catch (TrustKeyProcessingException e)
         {
            throw new GeneralSecurityException(e.getCause());
         }
         
         return RedirectBindingSignatureUtil.validateSignature(queryString, validatingKey, sigValue);
      }
      else
      {
         //Post binding no signature verification. The SAML message signature is verified
         return true;
      }
   }
   
   private void setRoleGenerator(String rgName)
   {
      try
      {
         Class<?> clazz = SecurityActions.getContextClassLoader().loadClass(rgName);
         rg = (RoleGenerator) clazz.newInstance();
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      } 
   }
}