/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.core.wstrust;

import java.security.Principal;
import java.security.PublicKey;

import org.jboss.identity.federation.core.wstrust.wrappers.RequestSecurityToken;
import org.jboss.identity.federation.ws.trust.RequestedReferenceType;
import org.jboss.identity.federation.ws.trust.StatusType;

/**
 * <p>
 * The {@code WSTrustRequestContext} contains all the information that is relevant for the security token request
 * processing. Its attributes are divided into two groups: attributes set by the request handler before calling a token
 * provider, and attributes set by the token provider after processing the token request.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class WSTrustRequestContext
{

   // information supplied by the request handler.
   private String tokenIssuer;

   private PublicKey providerPublicKey;

   private final Principal callerPrincipal;

   private final RequestSecurityToken request;

   // information supplied by the token provider.
   private SecurityToken securityToken;

   private StatusType status;

   private RequestedReferenceType attachedReference;

   private RequestedReferenceType unattachedReference;

   /**
    * <p>
    * Creates an instance of {@code WSTrustRequestContext} using the specified request.
    * </p>
    * 
    * @param request a {@code RequestSecurityToken} object that contains the information about the security token
    *            request.
    * @param callerPrincipal the {@code Principal} of the security token requester.
    */
   public WSTrustRequestContext(RequestSecurityToken request, Principal callerPrincipal)
   {
      this.request = request;
      this.callerPrincipal = callerPrincipal;
   }

   /**
    * <p>
    * Obtains the name of the token issuer (security token service name).
    * </p>
    * 
    * @return a {@code String} representing the token issuer name.
    */
   public String getTokenIssuer()
   {
      return tokenIssuer;
   }

   /**
    * <p>
    * Sets the name of the token issuer.
    * </p>
    * 
    * @param tokenIssuer a {@code String} representing the token issuer name.
    */
   public void setTokenIssuer(String tokenIssuer)
   {
      this.tokenIssuer = tokenIssuer;
   }

   /**
    * <p>
    * Obtains the {@code PublicKey} of the service provider that requires a security token.
    * </p>
    * 
    * @return the service provider's {@code PublicKey}.
    */
   public PublicKey getServiceProviderPublicKey()
   {
      return this.providerPublicKey;
   }

   /**
    * <p>
    * Sets the {@code PublicKey} of the service provider that requires a security token.
    * </p>
    * 
    * @param providerPublicKey the service provider's {@code PublicKey}.
    */
   public void setServiceProviderPublicKey(PublicKey providerPublicKey)
   {
      this.providerPublicKey = providerPublicKey;
   }

   /**
    * <p>
    * Obtains the principal of the WS-Trust token requester.
    * </p>
    * 
    * @return a reference to the caller {@code Principal} object.
    */
   public Principal getCallerPrincipal()
   {
      return this.callerPrincipal;
   }

   /**
    * <p>
    * Obtains the object the contains the information about the security token request.
    * </p>
    * 
    * @return a reference to the {@code RequestSecurityToken} instance.
    */
   public RequestSecurityToken getRequestSecurityToken()
   {
      return this.request;
   }

   /**
    * <p>
    * Obtains the security token contained in this context.
    * </p>
    * 
    * @return a reference to the {@code SecurityToken} instance.
    */
   public SecurityToken getSecurityToken()
   {
      return this.securityToken;
   }

   /**
    * <p>
    * Sets the security token in the context.
    * </p>
    * 
    * @param token the {@code SecurityToken} instance to be set.
    */
   public void setSecurityToken(SecurityToken token)
   {
      this.securityToken = token;
   }

   /**
    * <p>
    * Obtains the status of the security token validation.
    * </p>
    * 
    * @return a reference to the resulting {@code StatusType}.
    */
   public StatusType getStatus()
   {
      return this.status;
   }

   /**
    * <p>
    * Sets the status of the security token validation.
    * </p>
    * 
    * @param status a reference to the {@code StatusType} that represents the validation status.
    */
   public void setStatus(StatusType status)
   {
      this.status = status;
   }

   /**
    * <p>
    * Obtains the security token attached reference. This reference is used to locate the token inside the WS-Trust
    * response message when that token doesn't support references using URI fragments.
    * </p>
    * 
    * @return a {@code RequestedReferenceType} representing the attached reference.
    */
   public RequestedReferenceType getAttachedReference()
   {
      return this.attachedReference;
   }

   /**
    * <p>
    * Sets the security token attached reference. This reference is used to locate the token inside the WS-Trust
    * response message when that token doesn't support references using URI fragments.
    * </p>
    * 
    * @param attachedReference a {@code RequestedReferenceType} representing the attached reference.
    */
   public void setAttachedReference(RequestedReferenceType attachedReference)
   {
      this.attachedReference = attachedReference;
   }

   /**
    * <p>
    * Obtains the security token unattached reference. This reference is used to locate the token when it is not placed
    * inside the WS-Trust response message.
    * </p>
    * 
    * @return a {@code RequestedReferenceType} representing the unattached reference.
    */
   public RequestedReferenceType getUnattachedReference()
   {
      return this.unattachedReference;
   }

   /**
    * <p>
    * Sets the security token unattached reference. This reference is used to locate the token when it is not placed
    * inside the WS-Trust response message.
    * </p>
    * 
    * @param unattachedReference a {@code RequestedReferenceType} representing the unattached reference.
    */
   public void setUnattachedReference(RequestedReferenceType unattachedReference)
   {
      this.unattachedReference = unattachedReference;
   }
}
