/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.identity.federation.core.wstrust;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

/**
 * STSClientConfig has the ability to either programatically construct the configuration 
 * needed for {@link STSClient} or parse a file containing the configuration parameters.
 * <p/>
 * 
 * <h3>Configure programatically</h3>
 * Example:
 * <pre>{@code
 * Builder builder = new STSClientConfig.Builder();
 * builder.serviceName("JBossSTS");
 * builder.portName("JBossSTSPort");
 * ...
 * STSClientConfig config = builder.build();
 * }</pre>
 * 
 * <h3>Configure from file</h3>
 * Example:
 * <pre>{@code
 * STSClientConfig config = new STSClientConfig.Builder().build(configFile);
 * }</pre>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class STSClientConfig
{
    public static final String DEFAULT_CONFIG_FILE = "sts-client.properties";
    
    public static final String SERVICE_NAME = "serviceName";
    public static final String PORT_NAME = "portName";
    public static final String ENDPOINT_ADDRESS = "endpointAddress";
    public static final String USERNAME = "username";
    public static final String PASSWORD = "password";
    public static final String TOKEN_TYPE = "tokenType";

    private String serviceName;
    private String portName;
    private String endpointAddress;
    private String username;
    private String password;

    private STSClientConfig(final Builder builder)
    {
        serviceName = builder.serviceName;
        portName = builder.portName;
        endpointAddress = builder.endpointAddress;
        username = builder.username;
        password = builder.password;
    }

    public String getServiceName()
    {
        return serviceName;
    }

    public String getPortName()
    {
        return portName;
    }

    public String getEndPointAddress()
    {
        return endpointAddress;
    }

    public String getUsername()
    {
        return username;
    }

    public String getPassword()
    {
        return password;
    }
    
    public String toString()
    {
        return getClass().getSimpleName() + "[serviceName=" + serviceName + ", portName=" + portName + ", endpointAddress=" + endpointAddress + "]";
    }

    public static class Builder
    {
        private String serviceName;
        private String portName;
        private String endpointAddress;
        private String username;
        private String password;

        public Builder serviceName(final String serviceName)
        {
            this.serviceName = serviceName;
            return this;
        }

        public Builder portName(final String portName)
        {
            this.portName = portName;
            return this;
        }

        public Builder endpointAddress(final String address)
        {
            this.endpointAddress = address;
            return this;
        }

        public Builder username(final String username)
        {
            this.username = username;
            return this;
        }

        public Builder password(final String password)
        {
            this.password = password;
            return this;
        }
        
        public STSClientConfig build()
        {
            validate(this);
            return new STSClientConfig(this);
        }

        private void validate(Builder builder)
        {
            checkPropertyShowValue(serviceName, SERVICE_NAME);
            checkPropertyShowValue(portName, PORT_NAME);
            checkPropertyShowValue(endpointAddress, endpointAddress);
            checkProperty(username, USERNAME);
            checkProperty(password, PASSWORD);
        }

        private void checkPropertyShowValue(final String propertyName, final String propertyValue)
        {
            if (propertyValue == null || propertyValue.equals(""))
                throw new IllegalArgumentException(propertyName + " property must not be null or empty was:" + propertyValue);
        }

        private void checkProperty(final String propertyName, final String propertyValue)
        {
            if (propertyValue == null || propertyValue.equals(""))
                throw new IllegalArgumentException(propertyName + " property must not be null");
        }

        public STSClientConfig build(final String configFile)
        {
            InputStream in = null;

            try
            {
                in = getResource(configFile);
                if (in == null)
                {
                    throw new IllegalStateException("Could not find properties file " + configFile);
                    
                }
                final Properties properties = new Properties();
                properties.load(in);
                this.serviceName = properties.getProperty(SERVICE_NAME);
                this.portName = properties.getProperty(PORT_NAME);
                this.endpointAddress = properties.getProperty(ENDPOINT_ADDRESS);
                this.username = properties.getProperty(USERNAME);
                this.password = properties.getProperty(PASSWORD);
            }
            catch (IOException e)
            {
                throw new IllegalStateException("Could not load properties from " + configFile);
            }
            finally
            {
                try
                {
                    if (in != null)
                        in.close();
                }
                catch (final IOException ignored)
                {
                    ignored.printStackTrace();
                }
            }

            validate(this);
            return new STSClientConfig(this);
        }
    }

    private static InputStream getResource(String resource) throws IOException
    {
        // Try it as a File resource...
        final File file = new File(resource);

        if (file.exists() && !file.isDirectory())
        {
            return new FileInputStream(file);
        }
        // Try it as a classpath resource ...
        final ClassLoader threadClassLoader = Thread.currentThread().getContextClassLoader() ;
        if (threadClassLoader != null)
        {
            final InputStream is = threadClassLoader.getResourceAsStream(resource) ;
            if (is != null)
            {
                return is ;
            }
        }
        
        return  null;
    }

}

