/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.identity.federation.web.core;

import java.util.HashSet;
import java.util.Set;
import java.util.Stack;
import java.util.concurrent.ConcurrentHashMap;

import javax.servlet.http.HttpSessionEvent;
import javax.servlet.http.HttpSessionListener;

/**
 * Represents an Identity Server
 * @author Anil.Saldhana@redhat.com
 * @since Sep 17, 2009
 */
public class IdentityServer implements HttpSessionListener
{
   private STACK stack = new STACK(); 
   
   public class STACK
   {   
      private ConcurrentHashMap<String,Stack<String>> sessionParticipantsMap = 
         new ConcurrentHashMap<String, Stack<String>>();

      private ConcurrentHashMap<String, Set<String>> inTransitMap =
         new ConcurrentHashMap<String, Set<String>>();
      
      /**
       * Peek at the most recent participant in the session
       * @param sessionID
       * @return
       */
      public String peek(String sessionID)
      {
         Stack<String> stack = sessionParticipantsMap.get(sessionID);
         if(stack != null)
           return stack.peek();
         return "";
      }
      
      /**
       * Remove the most recent participant in the session
       * @param sessionID
       * @return
       */
      public String pop(String sessionID)
      {
         String result = null;
         Stack<String> stack = sessionParticipantsMap.get(sessionID);
         if(stack != null)
         {
            result = stack.pop(); 
         } 
         return result;
      }      

      /**
       * Register a participant in a session
       * @param sessionID
       * @param participant
       */
      public void register(String sessionID, String participant)
      {
         Stack<String> stack = sessionParticipantsMap.get(sessionID);
         if(stack.contains(participant) == false)
            stack.push(participant); 
      }

      /**
       * For a given identity session, return the number of participants
       * @param sessionID
       * @return
       */
      public int getParticipants(String sessionID)
      {
         Stack<String> stack = sessionParticipantsMap.get(sessionID);
         if(stack != null)
            return stack.size();
         
         return 0; 
      }
      
      /**
       * Register a participant as in transit in a logout interaction
       * @param sessionID
       * @param participant
       * @return
       */
      public boolean registerTransitParticipant(String sessionID, String participant)
      {
         Set<String> transitSet = inTransitMap.get(sessionID);
         if(transitSet != null)
            return transitSet.add(participant);
         return false;
      }
      
      /**
       * Deregister a participant as in transit in a logout interaction
       * @param sessionID
       * @param participant
       * @return
       */
      public boolean deRegisterTransitParticipant(String sessionID, String participant)
      {
         Set<String> transitSet = inTransitMap.get(sessionID);
         if(transitSet != null)
            return transitSet.remove(participant);
         return false;
      }
      
      /**
       * Return the number of participants in transit
       * @param sessionID
       * @return
       */
      public int getNumOfParticipantsInTransit(String sessionID)
      {
         Set<String> transitSet = inTransitMap.get(sessionID);
         if(transitSet != null)
            return transitSet.size();
         return 0; 
      }
      
      /**
       * The total number of sessions active
       * @return
       */
      public int totalSessions()
      {
         return sessionParticipantsMap.keySet().size();
      }
      
      private void put(String id)
      {
         sessionParticipantsMap.put(id, new Stack<String>());
         inTransitMap.put(id, new HashSet<String>());
      }
      
      private void remove(String id)
      {
         sessionParticipantsMap.remove(id);
         inTransitMap.remove(id);
      } 
   }
   
 
   /**
    * Return a reference to the internal stack 
    * @return
    */
   public STACK stack()
   {
      return stack;
   }
   

   /**
    * @see HttpSessionListener#sessionCreated(HttpSessionEvent)
    */
   public void sessionCreated(HttpSessionEvent sessionEvent)
   {   
      String id = sessionEvent.getSession().getId();
      stack.put(id); 
   }

   /**
    * @see HttpSessionListener#sessionDestroyed(HttpSessionEvent)
    */
   public void sessionDestroyed(HttpSessionEvent sessionEvent)
   {
      stack.remove(sessionEvent.getSession().getId()); 
   }
}