/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.metadata.spi.retrieval;

import org.jboss.logging.Logger;
import org.jboss.metadata.plugins.context.CachingMetaDataContext;
import org.jboss.metadata.spi.MetaData;
import org.jboss.metadata.spi.context.MetaDataContext;

/**
 * Utilities for MDR metadata
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class RetrievalUtils
{
   private final static Logger log = Logger.getLogger(RetrievalUtils.class);
   
   /**
    * Takes a MetaData and reworks the internal MetaDataContext to be caching.
    * If the metadata structure cannot be determined, a warning is logged, but no error is thrown.
    * 
    * @param metaData The meta data we want to cache
    * @return a cached meta data, or the original parameter if adding caching was not possible
    * @throws IllegalArgumentException if the metaData parameter is null
    */
   public static MetaData createCachedMetaData(MetaData metaData)
   {
      if (metaData == null)
         throw new IllegalArgumentException("Null meta data");
      
      if (metaData.getClass() != MetaDataRetrievalToMetaDataBridge.class)
      {
         log.warn("MetaDataUtils.createCachedMetaData(): meta data is not a MetaDataRetrievalToMetaDataBridge, not adding caching");
         return metaData;
      }
       
      if (((MetaDataRetrievalToMetaDataBridge)metaData).getMetaDataRetrieval() instanceof MetaDataContext == false)
      {
         log.warn("MetaDataUtils.createCachedMetaData(): Could not determine structure of meta data, not adding caching");
         return metaData;
      }
      
      MetaDataContext ctx = (MetaDataContext)((MetaDataRetrievalToMetaDataBridge)metaData).getMetaDataRetrieval();
      return new MetaDataRetrievalToMetaDataBridge(createCachedMetaDataContext(ctx));
   }

   /**
    * Creates a cached meta data context for the passed in context. All the parent contexts
    * will also be cached
    * 
    * @param ctx the meta data context we want to cache
    * @param the cached meta data context
    */
   private static MetaDataContext createCachedMetaDataContext(MetaDataContext ctx)
   {
      MetaDataContext parent = ctx.getParent();
      if (parent != null)
      {
         parent = createCachedMetaDataContext(parent);
      }
      return new CachingMetaDataContext(parent, ctx.getLocalRetrievals());
   }
}
