/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins;

import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.util.HashSet;
import java.util.Set;

/**
 * Common methods needed by both the javassist and introspection implementations 
 * for figuring out generics.
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class GenericsUtil
{
   /**
    * Create the name of the generic type
    * 
    *  @param type the generic type
    *  @return the name
    */
   public static String getGenericName(Type type)
   {
      StringBuilder sb = new StringBuilder();
      appendTypeGenericInfo(type, sb, null);
      return sb.toString();
   }

   protected static void appendTypeGenericInfo(Type type, StringBuilder sb, Set<String> doneVars)
   {
      if (type instanceof Class)
      {
         sb.append(((Class<?>)type).getName());
      }
      else if (type instanceof WildcardType)
      {
         WildcardType wtype = (WildcardType)type;
         Type[] lower = wtype.getLowerBounds();
         Type[] upper = wtype.getUpperBounds();
         
         if (lower.length > 0)
         {
            appendTypeGenericInfo(lower[0], sb, doneVars);
         }
         else if (upper.length > 0)
         {
            appendTypeGenericInfo(upper[0], sb, doneVars);
         }
      }
      else if (type instanceof ParameterizedType)
      {
         ParameterizedType ptype = (ParameterizedType)type;
         appendTypeGenericInfo(ptype.getRawType(), sb, doneVars);
         StringBuilder params = new StringBuilder();
         params.append("<");
         Type[] types = ptype.getActualTypeArguments();
         for (int i = 0 ; i < types.length ; i++)
         {
            if (i > 0)
               params.append(", ");
            appendTypeGenericInfo(types[i], params, doneVars);
         }
         if (params.length() > 1)
         {
            params.append(">");
            sb.append(params.toString());
         }
      }
      else if (type instanceof GenericArrayType)
      {
         GenericArrayType gtype = (GenericArrayType)type;
         appendTypeGenericInfo(gtype.getGenericComponentType(), sb, doneVars);
         sb.append("[]");
      }
      else if (type instanceof TypeVariable)
      {
         TypeVariable typeVar = (TypeVariable)type;
         if (doneVars == null)
            doneVars = new HashSet<String>();
         if (!doneVars.contains(typeVar.getName()))
         {
            doneVars.add(typeVar.getName());
            appendTypeGenericInfo(typeVar.getBounds()[0], sb, doneVars);
         }
      }
      else
      {
         //TODO This might need implementing once we test wildcards
         throw new IllegalArgumentException("Unhandled type " + type.getClass().getName());
      }
   }
   
   public static ClassLoader findClassLoader(Type type)
   {
      if (type instanceof Class)
         return SecurityActions.getClassLoader((Class<?>)type);
      if (type instanceof ParameterizedType)
      {
         return handleNullClassLoader(findClassLoader(((ParameterizedType)type).getRawType()));
      }
      if (type instanceof WildcardType)
      {
         WildcardType wtype = (WildcardType)type;
         if (wtype.getUpperBounds().length > 0)
            return handleNullClassLoader(findClassLoader(wtype.getUpperBounds()[0]));
         if (wtype.getLowerBounds().length > 0)
            return handleNullClassLoader(findClassLoader(wtype.getLowerBounds()[0]));
      }
      if (type instanceof GenericArrayType)
      {
         return handleNullClassLoader(findClassLoader(((GenericArrayType)type).getGenericComponentType()));
      }
      if (type instanceof TypeVariable)
      {
         TypeVariable<?> typeVar = (TypeVariable<?>)type;
         return handleNullClassLoader(findClassLoader(typeVar.getBounds()[0]));
      }
      
      throw new IllegalArgumentException(type.getClass().getName() + " is not handled yet");
   }
   
   private static ClassLoader handleNullClassLoader(ClassLoader loader)
   {
      if (loader != null)
         return loader;
      
      return SecurityActions.getSystemClassLoader();
   }

}
