/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode;

import javassist.bytecode.SignatureAttribute.ClassSignature;
import javassist.bytecode.SignatureAttribute.MethodSignature;
import javassist.bytecode.SignatureAttribute.Type;
import javassist.bytecode.SignatureAttribute.TypeParameter;
import javassist.bytecode.SignatureAttribute.TypeVariable;

/**
 * When a type is of type {@link TypeVariable} this class helps determine what the actual type is
 * from the signature of the member or the class
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
abstract class BytecodeTypeVariableSpy
{
   /**
    * Create a JavassistTypeVariableSpy for a method or constructor
    * 
    * @param classSig the signature of the declaring class
    * @param sig the method signature
    * @return the JavassistTypeVariableSpy
    */
   static BytecodeTypeVariableSpy createForBehavior(ClassSignature classSig, MethodSignature sig)
   {
      return new MethodTypeVariable(classSig, sig);
   }
   
   /**
    * Create a JavassistTypeVariableSpy for a class
    * 
    * @param sig the class signature of the class
    * @return the JavassistTypeVariableSpy
    */
   static BytecodeTypeVariableSpy createForClass(ClassSignature sig)
   {
      return new ClassTypeVariable(sig);
   }
   
   /**
    * Create a JavassistTypeVariableSpy for a field
    * 
    * @param sig the class signature of the class containing the field
    * @return the JavassistTypeVariableSpy
    */
   static BytecodeTypeVariableSpy createForField(ClassSignature sig)
   {
      return new ClassTypeVariable(sig);
   }
   
   /**
    * Get the actual bound of a type variable
    * 
    * @param tv the type variable
    * @throws IllegalStateException if the type can not be determined. This should only 
    * happen if the TypeVariable passed in does not belong to the signature used to 
    * create this JavassistTypeVariableSpy instance
    */
   abstract Type getTypeBound(TypeVariable tv);
   
   private static Type getTypeFromTypeParameters(TypeVariable tv, TypeParameter[] parameters)
   {
      if (parameters == null)
         return null;

      for (int i = 0 ; i < parameters.length ; i++)
      {
         if (parameters[i].getName().equals(tv.getName()))
         {
            Type type = parameters[i].getClassBound();
            if (type != null)
               return type;
            Type[] types = parameters[i].getInterfaceBound();
            if (types != null && types.length > 0)
               return types[0];
            
            break;
         }
      }
      
      return null;
   }
   
   private static class MethodTypeVariable extends BytecodeTypeVariableSpy
   {
      protected final ClassSignature classSig;
      private final MethodSignature sig;

      MethodTypeVariable(ClassSignature classSig, MethodSignature sig)
      {
         this.classSig = classSig;
         this.sig = sig;
      }

      @Override
      Type getTypeBound(TypeVariable tv)
      {
         Type type = getTypeFromTypeParameters(tv, sig.getTypeParameters());
         if (type != null)
            return type;

         type = getTypeFromTypeParameters(tv, classSig.getParameters());
         if (type != null)
            return type;
         
         throw new IllegalStateException("No type parameter found called " + tv.getName() + " in " + sig + " or in " + classSig);
      }
   }
   
   private static class ClassTypeVariable extends BytecodeTypeVariableSpy
   {
      private final ClassSignature sig;

      public ClassTypeVariable(ClassSignature sig)
      {
         this.sig = sig;
      }

      @Override
      Type getTypeBound(TypeVariable tv)
      {
         Type type = getTypeFromTypeParameters(tv, sig.getParameters());
         if (type != null)
            return type;

         throw new IllegalStateException("No type parameter found called " + tv.getName() + " in " + sig);
      }
   }   
}
