/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode.accessor.metrics;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class Metrics implements MetricsMBean
{
   final static List<Entry> record = new ArrayList<Entry>();
   
   final Entry entry;

   volatile boolean locked;
   
   public Metrics()
   {
      entry = null;
   }
   
   protected Metrics(String name)
   {
      if (name == null)
         throw new IllegalArgumentException("Null name");
      this.entry = new Entry(name);
      synchronized (record)
      {
         record.add(entry);
      }
   }
   
   static String getDelegateType(Object delegate)
   {
      if (delegate.getClass().getSimpleName().startsWith("Default"))
         return " - {R}"; 
      
      return " - {G}";
   }

   protected void recordTime(long nanosTaken)
   {
      if (locked)
      {
         //This is not perfect, but I don't want to synchronize all the time
         synchronized (record)
         {
            doRecordTime(nanosTaken);
         }
      }
      doRecordTime(nanosTaken);
   }
   
   protected void doRecordTime(long nanosTaken)
   {
      entry.getInvocations().incrementAndGet();
      entry.getNanos().addAndGet(nanosTaken);
   }
   
   public List<Entry> sortByInvocations()
   {
      return sort(InvocationsComparator.INSTANCE);
   }
   
   public String outputEntriesByInvocations(String lineSeparator)
   {
      if (lineSeparator == null)
         lineSeparator = "\n";
      
      StringBuilder sb = new StringBuilder("Invocations (total time in ms) - Member name");
      sb.append(lineSeparator);
      sb.append("============================================");
      sb.append(lineSeparator);
      
      long totalTime =0, invocations = 0;
      for (Entry entry : sortByInvocations())
      {
         totalTime += entry.getNanos().get();
         invocations += entry.getInvocations().get();
         
         sb.append(entry.getInvocations() + " (" + entry.getNanos().get()/*/1000000*/ + ") - " + entry.getName());
         sb.append(lineSeparator);
      }
      sb.append("--------------------------------------------");
      sb.append(lineSeparator);
      sb.append("Total: " + invocations + " (" + totalTime + ")");
      
      return sb.toString();
   }

   public List<Entry> sortByTime()
   {
      return sort(TimeComparator.INSTANCE);
   }
   
   public String outputEntriesByTime(String lineSeparator)
   {
      if (lineSeparator == null)
         lineSeparator = "\n";
      
      StringBuilder sb = new StringBuilder("Total time in ms (Invocations) - Member name");
      sb.append(lineSeparator);
      sb.append("=============================================");
      sb.append(lineSeparator);
      
      long totalTime =0, invocations = 0;
      for (Entry entry : sortByTime())
      {
         totalTime += entry.getNanos().get();
         invocations += entry.getInvocations().get();
         
         sb.append(entry.getNanos().get()/*/1000000*/ + " (" + entry.getInvocations() + ") - " + entry.getName());
         sb.append(lineSeparator);
      }
      
      sb.append("--------------------------------------------");
      sb.append(lineSeparator);
      sb.append("Total: " + totalTime  + " (" + invocations + ")");
      return sb.toString();
   }
   
   private List<Entry> sort(Comparator<Entry> comparator)
   {
      synchronized (record)
      {
         try
         {
            locked = true;
            List<Entry> entries = new ArrayList<Entry>(record);
            Collections.sort(entries, comparator);
            return entries;
         }
         finally
         {
            locked = false;
         }
      }
   }

   private static class InvocationsComparator implements Comparator<Entry>
   {
      final static InvocationsComparator INSTANCE = new InvocationsComparator();
      
      public int compare(Entry o1, Entry o2)
      {
         if (o1.getInvocations().get() == o2.getInvocations().get())
            return 0;
         else if (o1.getInvocations().get() > o2.getInvocations().get())
            return -1;
         return 1;
      }
   }
   
   private static class TimeComparator implements Comparator<Entry>
   {
      final static TimeComparator INSTANCE = new TimeComparator();
      
      public int compare(Entry o1, Entry o2)
      {
         if (o1.getNanos().get() == o2.getNanos().get())
            return 0;
         else if (o1.getNanos().get() > o2.getNanos().get())
            return -1;
         return 1;
      }
   }
   
   public static class Entry
   {
      final String name;
      final AtomicInteger invocations = new AtomicInteger();
      final AtomicLong nanos = new AtomicLong();
      
      public Entry(String name)
      {
         this.name = name;
      }
      
      public String getName()
      {
         return name;
      }

      public AtomicInteger getInvocations()
      {
         return invocations;
      }

      public AtomicLong getNanos()
      {
         return nanos;
      }
   }
}
