/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode.bytes.asm;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.lang.ref.WeakReference;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.jboss.reflect.plugins.bytecode.bytes.ClassBytes;
import org.jboss.reflect.plugins.bytecode.bytes.ConstructorBytes;
import org.jboss.reflect.plugins.bytecode.bytes.FieldBytes;
import org.jboss.reflect.plugins.bytecode.bytes.MethodBytes;
import org.jboss.reflect.plugins.bytecode.bytes.PrimitiveBytes;
import org.jboss.reflect.util.objectweb.asm.AnnotationVisitor;
import org.jboss.reflect.util.objectweb.asm.ClassReader;
import org.jboss.reflect.util.objectweb.asm.FieldVisitor;
import org.jboss.reflect.util.objectweb.asm.MethodVisitor;
import org.jboss.reflect.util.objectweb.asm.Opcodes;
import org.jboss.reflect.util.objectweb.asm.commons.EmptyVisitor;
import org.jboss.util.JBossObject;
import org.jboss.util.JBossStringBuilder;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class AsmClassBytes extends JBossObject implements ClassBytes
{
   static final int STANDARD_FLAGS = ClassReader.SKIP_CODE | ClassReader.SKIP_DEBUG | ClassReader.SKIP_FRAMES 
      | ClassReader.SKIP_OUTER_CLASSES | ClassReader.SKIP_UNKNOWN_ATTRIBUTES;
   
   static final int READ_FIELD_FLAGS = STANDARD_FLAGS | ClassReader.INCLUDE_FIELD_OVERVIEW;
   
   static final ClassReaderCacheFactory CACHE_FACTORY = ClassReaderCacheFactory.createFromProperties();
   
   static final FieldBytes[] NO_FIELDS = new FieldBytes[0];
   
   static final MethodBytes[] NO_METHODS = new MethodBytes[0];
   
   static final ConstructorBytes[] NO_CONSTRUCTORS = new ConstructorBytes[0];
   
   static final Annotation[] NO_ANNOTATIONS = new Annotation[0];

   //TODO some magic timeout cache so we don't keep the full class bytes in memory
   private final ClassReaderHolder readerHolder;

   private WeakReference<ClassLoader> loader;
   
   private final URL url;
   
   private final String jvmName;
   
   private volatile int modifiers = Integer.MIN_VALUE;
   
   private volatile ConstructorBytes[] constructorBytes;
   
   private volatile MethodBytes[] methodBytes;
   
   private volatile FieldBytes[] fieldBytes;
   
   public AsmClassBytes(ClassLoader loader, URL url, String name)
   {
      this.loader = new WeakReference<ClassLoader>(loader);
      this.jvmName = name;
      this.url = url;
      readerHolder = CACHE_FACTORY == null ? new HardReferenceClassReaderHolder() : new CachingClassReaderHolder(); 
   }

   ClassReader getReader()
   {
      return readerHolder.getClassReader();
   }
   
   public ConstructorBytes[] getDeclaredConstructorBytes()
   {
      if (constructorBytes == null)
         initMethodsAndConstructors();
      return constructorBytes;
   }

   public FieldBytes[] getDeclaredFieldBytes()
   {
      if (fieldBytes == null)
      {
         FieldsVisitor visitor = new FieldsVisitor();
         getReader().accept(visitor, STANDARD_FLAGS | ClassReader.INCLUDE_FIELD_OVERVIEW);
         fieldBytes = visitor.getFieldBytes();
      }
      return fieldBytes;
   }

   public MethodBytes[] getDeclaredMethodBytes()
   {
      if (methodBytes == null)
         initMethodsAndConstructors();
      return methodBytes;
   }

   public String[] getInterfaceJvmNames()
   {
      return getReader().getInterfaces();
   }

   public String[] getInterfaceTypeInfoNames()
   {
      String[] jvmNames = getInterfaceJvmNames();
      if (jvmNames == null || jvmNames.length == 0)
         return jvmNames;
      
      String[] typeInfoNames = new String[jvmNames.length];

      for (int i = 0; i < typeInfoNames.length; i++)
         typeInfoNames[i] = Util.jvmNameToTypeInfoName(jvmNames[i]);
         
      return typeInfoNames;
   }
   public String getSuperClassJvmName()
   {
      return getReader().getSuperName();
   }
   
   public String getSuperClassTypeInfoName()
   {
      String superName = getSuperClassJvmName();
      if (superName == null)
         return null;
      return Util.jvmNameToTypeInfoName(superName);
   }

   public boolean isAnnotation()
   {
      return (getModifiers() & Opcodes.ACC_ANNOTATION) != 0;
   }

   public boolean isArray()
   {
      return false;
   }

   public boolean isEnum()
   {
      return (getModifiers() & Opcodes.ACC_ENUM) != 0;
     
   }

   public boolean isInterface()
   {
      return (getModifiers() & Opcodes.ACC_INTERFACE) != 0;
   }

   
   public int getModifiers()
   {
      if (modifiers != Integer.MIN_VALUE)
         return modifiers;
      
      int mods = getReader().getAccess();
      //We sometimes get the 'super' flag set and that should not be there
      if ((mods & Opcodes.ACC_SUPER) == Opcodes.ACC_SUPER)
         mods = mods ^ Opcodes.ACC_SUPER;
      modifiers = mods;
      return modifiers;
   }

   public String getJvmName()
   {
      return jvmName;
   }
   
   public String getTypeInfoName()
   {
      return Util.jvmNameToTypeInfoName(jvmName);
   }

   public ClassLoader getClassLoader()
   {
      return loader.get();
   }
   
   public String getGenericSignature()
   {
      ClassSignatureVisitor visitor = new ClassSignatureVisitor();
      getReader().accept(visitor, STANDARD_FLAGS | ClassReader.INCLUDE_HEADER);      
      return visitor.getSignature();
   }
   
   public Annotation[] getAnnotations()
   {
      AnnotationsVisitor visitor = new AnnotationsVisitor();
      getReader().accept(visitor, STANDARD_FLAGS | ClassReader.INCLUDE_CLASS_ANNOTATIONS);
      return visitor.getAnnotations();
   }

   private void initMethodsAndConstructors()
   {
      ConstructorAndMethodsVisitor visitor = new ConstructorAndMethodsVisitor();
      getReader().accept(visitor, STANDARD_FLAGS | ClassReader.INCLUDE_METHOD_OVERVIEW);
      methodBytes = visitor.getMethodBytes();
      constructorBytes = visitor.getConstructorBytes();
   }
   
   public ClassBytes getComponentType()
   {
      return null;
   }

   private static class ClassSignatureVisitor extends EmptyVisitor
   {
      String signature;

      @Override
      public void visit(int version, int access, String name, String signature, String superName, String[] interfaces)
      {
         this.signature = signature;
      }
      
      public String getSignature()
      {
         return signature;
      }
   }
   
   private class AnnotationsVisitor extends Util.EmptyClassVisitor implements Util.ParentReader
   {
      List<Annotation> annotations;
      public AnnotationVisitor visitAnnotation(String desc, boolean visible)
      {
         return Util.createAnnotationVisitor(this, getClassLoader(), desc);
      }
      
      public Annotation[] getAnnotations()
      {
         if (annotations == null)
            return NO_ANNOTATIONS;
         return annotations.toArray(new Annotation[0]);
      }

      public void setValueInParent(String name, Object value)
      {
         if (annotations == null)
            annotations = new ArrayList<Annotation>();
         annotations.add((Annotation)value);
      }
   }
   
   private class ConstructorAndMethodsVisitor extends EmptyVisitor
   {
      List<ConstructorBytes> constructorBytes;
      List<MethodBytes> methodBytes;
      
      @Override
      public MethodVisitor visitMethod(int access, String name, String desc, String signature, String[] exceptions, int byteCodeIndex)
      {
         if (name.startsWith("<c")) //<clinit>
            return null;
         if (name.startsWith("<i")) //<init>
         {
            if (constructorBytes == null)
               constructorBytes = new ArrayList<ConstructorBytes>();
            constructorBytes.add(new AsmConstructorBytes(AsmClassBytes.this, access, name, desc, signature, exceptions, byteCodeIndex));
         }
         else
         {
            if (methodBytes == null)
               methodBytes = new ArrayList<MethodBytes>();
            methodBytes.add(new AsmMethodBytes(AsmClassBytes.this, access, name, desc, signature, exceptions, byteCodeIndex));
         }
         return null;
      }
      
      ConstructorBytes[] getConstructorBytes()
      {
         if (constructorBytes == null)
            return NO_CONSTRUCTORS;
         return constructorBytes.toArray(new ConstructorBytes[constructorBytes.size()]);
      }
      
      
      MethodBytes[] getMethodBytes()
      {
         if (methodBytes == null)
            return NO_METHODS;
         return methodBytes.toArray(new MethodBytes[methodBytes.size()]);
      }
   }
   
   private class FieldsVisitor extends EmptyVisitor
   {
      List<FieldBytes> fieldBytes;

      @Override
      public FieldVisitor visitField(int access, String name, String desc, String signature, Object value, int index)
      {
         if (fieldBytes == null)
            fieldBytes =  new ArrayList<FieldBytes>();
         fieldBytes.add(new AsmFieldBytes(AsmClassBytes.this, access, name, desc, signature, index));
         return null;
      }
      
      FieldBytes[] getFieldBytes()
      {
         if (fieldBytes == null)
            return NO_FIELDS;
         return fieldBytes.toArray(new FieldBytes[fieldBytes.size()]);
      }
   }
   
   private interface ClassReaderHolder
   {
      ClassReader getClassReader();
   }
   
   private abstract class AbstractClassReaderHolder implements ClassReaderHolder  
   {
      protected ClassReader loadClassReader()
      {
         InputStream in = null;
         try
         {
            in = new BufferedInputStream(url.openStream());
            return new ClassReader(in);
         }
         catch (IOException e)
         {
            throw new RuntimeException(e);
         }
         finally 
         {
            try
            {
               if (in != null)
                  in.close();
            }
            catch (IOException ignoree)
            {
            }
         }
      }
   }
   
   private class HardReferenceClassReaderHolder extends AbstractClassReaderHolder
   {
      volatile ClassReader reader;
      
      public ClassReader getClassReader()
      {
         if (reader == null)
            reader = loadClassReader();
         return reader;
      }
   }
   
   private class CachingClassReaderHolder extends AbstractClassReaderHolder
   {
      final ClassReaderCache cache;

      public CachingClassReaderHolder()
      {
         cache = CACHE_FACTORY.createCache();
      }
      
      public ClassReader getClassReader()
      {
         ClassReader reader = cache.get(jvmName);
         if (reader == null)
         {
            reader = loadClassReader();
            cache.put(jvmName, reader);
         }
         return reader;
      }
   }

   public PrimitiveBytes getPrimitive()
   {
      return null;
   }

   @Override
   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(getJvmName());
   }

   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      toShortString(buffer);
   }
}
