/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist.bytecode;

import org.jboss.reflect.plugins.javassist.JavassistAnnotatedParameterInfo;
import org.jboss.reflect.plugins.javassist.JavassistConstructor;
import org.jboss.reflect.plugins.javassist.JavassistMethod;
import org.jboss.reflect.plugins.javassist.SignatureKey;
import org.jboss.reflect.spi.TypeInfo;

/**
 * <p>Abstract class to create implementations of the {@link JavassistMethod} and {@link JavassistConstructor}
 * interfaces.</p>
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public abstract class JavassistBehaviorFactory extends JavassistMemberFactory
{
   /** The constructor or method being targeted */
   JavassistAnnotatedParameterInfo behavior;
   
   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param behavior the method or constructor we are generating a {@link JavassistMethod} or {@link JavassistConstructor} for
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   JavassistBehaviorFactory(Class<?> superClass, JavassistAnnotatedParameterInfo behavior, boolean debug)
   {
      super(superClass, debug);
      this.behavior = behavior;
   }
   
   /**
    * Gets the parameter types of the method or constructor we are generating a {@link JavassistMethod} or {@link JavassistConstructor} for.
    * 
    * @return the parameter types
    */
   TypeInfo[] getParameterTypes()
   {
      return behavior.getParameterTypes();
   }
   
   @Override
   String initAccessedMember()
   {
      return behavior.getDeclaringClass().getName() + "." + behavior.getName() + behavior.getDescriptor();
   }

   /**
    * Adds the byte code to push the parameters for a method/constructor 
    * on to the stack
    * 
    * @param cfwc the byte code of the method we are generating
    * @param key the target parameter types
    * @param the index of the parameter array
    */
   void addParameters(ClassFileWriterContext<?> cfwc, SignatureKey key, int parameterIndex)
   {
      String[] params = key.getParams();
      for (int i = 0 ; i < params.length ; i++)
      {
         cfwc.addAload(parameterIndex);
         cfwc.addIconst(i);
         cfwc.addAALoad();
         
         if (!params[i].equals(OBJECT_NAME))
         {
            castAndUnbox(cfwc, params[i]);
         }
      }
   }
   
   JavassistAnnotatedParameterInfo getBehavior()
   {
      return behavior;
   }
}
