/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.bytecode;

import java.lang.reflect.Modifier;

import javassist.bytecode.BadBytecode;
import javassist.bytecode.SignatureAttribute;
import javassist.bytecode.SignatureAttribute.ClassSignature;
import javassist.bytecode.SignatureAttribute.ObjectType;

import org.jboss.reflect.plugins.AnnotationHelper;
import org.jboss.reflect.plugins.bytecode.accessor.FieldAccessor;
import org.jboss.reflect.plugins.bytecode.accessor.MemberAccessorFactory;
import org.jboss.reflect.plugins.bytecode.bytes.FieldBytes;
import org.jboss.reflect.spi.AnnotationValue;
import org.jboss.reflect.spi.FieldInfo;
import org.jboss.reflect.spi.MutableFieldInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.JBossStringBuilder;

/**
 * FieldInfo that relies on Javassist to answer reflective queries and to access
 * the represented field.
 * 
 *  This class also relies on Javassist to perform the instrumentation operations defined in
 *  {@code MutableFieldInfo}.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 105060 $
 * @see MutableFieldInfo
 */
public class BytecodeFieldInfo extends BytecodeAnnotatedInfo implements FieldInfo
{
   /** The serialVersionUID */
   private static final long serialVersionUID = -104555531831318930L;

   /** The field */
   private final FieldBytes fieldBytes;
   
   /** The field implementation */
   private transient volatile FieldAccessor field;

   /** The type */
   private transient volatile TypeInfo fieldType;
   
   /** The type info */
   protected final BytecodeTypeInfo typeInfo;

   private transient volatile String name;
   
   private transient volatile boolean initializedModifiers;
   
   private transient volatile int modifiers;
   
   /**
    * Create a new JavassistFieldInfo.
    * 
    * @param annotationHelper the annotation helper
    * @param typeInfo the type info
    * @param ctField the field
    */
   public BytecodeFieldInfo(AnnotationHelper annotationHelper, BytecodeTypeInfo typeInfo, FieldBytes fieldBytes)
   {
      super(annotationHelper);
      this.typeInfo = typeInfo;
      this.fieldBytes = fieldBytes;
   }

   public String getName()
   {
      if (name == null)
         name = fieldBytes.getName();
      return name;
   }

   public int getModifiers()
   {
      if (!initializedModifiers)
      {
         modifiers = fieldBytes.getModifiers();
         initializedModifiers = true;
      }
     
      return modifiers;
   }

   public boolean isPublic()
   {
      return Modifier.isPublic(getModifiers());
   }

   public boolean isStatic()
   {
      return Modifier.isStatic(getModifiers());
   }

   public boolean isVolatile()
   {
      return Modifier.isVolatile(getModifiers());
   }

   public BytecodeTypeInfo getDeclaringClass()
   {
      return typeInfo;
   }

   public TypeInfo getType()
   {
      if (fieldType != null)
         return fieldType;
      String signature = fieldBytes.getGenericSignature();
      if (signature != null)
      {
         ObjectType type;
         try
         {
            type = SignatureAttribute.toFieldSignature(signature);
         }
         catch (BadBytecode e)
         {
            throw new RuntimeException(e);
         }
         ClassSignature sig = getDeclaringClass().getClassSignature();
         fieldType = typeInfo.getFactory().getTypeInfo(typeInfo.getClassLoaderInternal(), type, BytecodeTypeVariableSpy.createForField(sig));
      }
      else
      {
         try
         {
            fieldType = typeInfo.getFactory().getTypeInfo(SignatureKey.getFieldType(fieldBytes), getDeclaringClass().getClassLoaderInternal());
         }
         catch (ClassNotFoundException e)
         {
            throw new RuntimeException(e);
         }
      }
      return fieldType;
   }

   public Object get(Object target) throws Throwable
   {
      BytecodeAccessController.checkAccess(this);
      
      return getJavassistField().get(target);
   }

   public Object set(Object target, Object value) throws Throwable
   {
      BytecodeAccessController.checkAccess(this);

      getJavassistField().set(target, value);
      return null;
   }

   private FieldAccessor getJavassistField() throws Throwable
   {
      if (field == null)
      {
         FieldAccessor f = MemberAccessorFactory.INSTANCE.createField(this);
         if (field == null)
            field = f;
      }
      return field;
   }
   
   @Override
   protected int getHashCode()
   {
      return getName().hashCode();
   }

   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof FieldInfo == false)
         return false;
      
      FieldInfo other = (FieldInfo) obj;
      if (getName().equals(other.getName()) == false)
         return false;
      return getDeclaringClass().equals(other.getDeclaringClass());
   }

   @Override
   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(getName());
   }

   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      buffer.append("name=").append(getName());
      super.toString(buffer);
   }
   
   public AnnotationValue[] getAnnotations()
   {
      return getAnnotations(fieldBytes);
   }
   
   public String getDescriptor()
   {
      return fieldBytes.getJvmSignature();
   }
}
