/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.bytecode;

import java.util.Arrays;

import javassist.bytecode.SignatureAttribute.BaseType;
import javassist.bytecode.SignatureAttribute.MethodSignature;

import org.jboss.reflect.plugins.AnnotationHelper;
import org.jboss.reflect.plugins.MethodInfoInternal;
import org.jboss.reflect.plugins.bytecode.accessor.MemberAccessorFactory;
import org.jboss.reflect.plugins.bytecode.accessor.MethodAccessor;
import org.jboss.reflect.plugins.bytecode.bytes.MethodBytes;
import org.jboss.reflect.spi.MethodInfo;
import org.jboss.reflect.spi.MutableMethodInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.JBossStringBuilder;


/**
 * MethodInfo that relies on Javassist to answer reflective queries and to invoke
 * the represented method.
 * 
 *  This class also relies on Javassist to perform the instrumentation operations defined in
 *  {@code MutableMethodInfo}.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 105101 $
 * @see MutableMethodInfo
 */
public class BytecodeMethodInfo extends BytecodeBehaviourInfo implements MethodInfo, MethodInfoInternal
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 101183748227690112L;

   /** The method implementation */
   private transient volatile MethodAccessor method;
   
   /** The return type */
   private transient volatile TypeInfo returnType;
   
   /**
    * Create a new JavassistMethodInfo.
    * 
    * @param annotationHelper the annotation helper
    * @param typeInfo the type info
    * @param method the method
    */
   public BytecodeMethodInfo(AnnotationHelper annotationHelper, BytecodeTypeInfo typeInfo, MethodBytes method)
   {
      super(annotationHelper, typeInfo, method);
   }

   public String getName()
   {
      return behaviour.getName();
   }

   public int getNumberParameters()
   {
      return getSignatureKey().getParams().length;
   }
   
   public TypeInfo getReturnType()
   {
      if (returnType != null)
         return returnType;

      MethodSignature sig = getMethodSignature();
      boolean doneReturn = false;
      if (sig != null)
      {
         if (sig.getReturnType() instanceof BaseType == false)
         {
            returnType = typeInfo.getFactory().getTypeInfo(typeInfo.getClassLoaderInternal(), sig.getReturnType(), BytecodeTypeVariableSpy.createForBehavior(((BytecodeClassInfo)getDeclaringClass()).getClassSignature(), sig));
            doneReturn = true;
         }
      }

      if (!doneReturn)
      {
         String name = SignatureKey.getReturnType(behaviour);
         try
         {
            returnType = typeInfo.getFactory().getTypeInfo(name, typeInfo.getClassLoaderInternal());
         }
         catch (ClassNotFoundException e)
         {
            throw new RuntimeException(e);
         }
      }
      return returnType;
   }

   public Object invoke(Object target, Object[] args) throws Throwable
   {
      if (method == null)
      {
         MethodAccessor m = MemberAccessorFactory.INSTANCE.createMethod(this);
         if (method == null)
            method = m;
      }
      
      BytecodeAccessController.checkAccess(this);
      
      return method.invoke(target, args);
   }

   @Override
   protected int getHashCode()
   {
      return getName().hashCode();
   }

   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof MethodInfo == false)
         return false;

      final MethodInfo other = (MethodInfo) obj;

      if (getName().equals(other.getName()) == false)
         return false;
//      if (getDeclaringClass().equals(other.getDeclaringClass()) == false)
//         return false;
//      if (getReturnType().equals(other.getReturnType()) == false)
//         return false;
      
      //Check the signature key since we want to avoid calling getParameterTypes(). If the parameters have
      //not already been loaded that means hitting the classpools which is costly and might not be needed
      //for anything else
      if (other instanceof BytecodeMethodInfo)
         return getSignatureKey().equals(((BytecodeMethodInfo)(other)).getSignatureKey());
      
      return Arrays.equals(getParameterTypes(), other.getParameterTypes());
   }

   @Override
   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(getName());
   }

   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      buffer.append("name=").append(getName());
      super.toString(buffer);
   }
}
