/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode.bytes.asm;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.List;

import org.jboss.reflect.plugins.bytecode.bytes.BehaviourBytes;
import org.jboss.reflect.util.objectweb.asm.AnnotationVisitor;
import org.jboss.reflect.util.objectweb.asm.Attribute;
import org.jboss.reflect.util.objectweb.asm.ClassReader;
import org.jboss.reflect.util.objectweb.asm.MethodVisitor;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
abstract class AsmBehaviourBytes extends AsmMemberBytes implements BehaviourBytes
{
   final static String[] NO_EXCEPTIONS = new String[0];
   
   private final String[] exceptions;

   public AsmBehaviourBytes(AsmClassBytes clazz, int modifiers, String name, String desc, String signature, String[] exceptions, int byteCodeIndex)
   {
      super(clazz, modifiers, name, desc, signature, byteCodeIndex);
      
      this.exceptions = exceptions != null ? exceptions : NO_EXCEPTIONS;
   }


   public String[] getExceptionTypeJvmNames()
   {
      return exceptions;
   }
   
   public String[] getExceptionTypeTypeInfoNames()
   {
      String[] jvmExceptions = getExceptionTypeJvmNames();
      if (jvmExceptions.length == 0)
         return NO_EXCEPTIONS;
      
      String[] typeInfoExceptions = new String[jvmExceptions.length];
      for (int i = 0 ; i < jvmExceptions.length ; i++)
         typeInfoExceptions[i] = Util.jvmNameToTypeInfoName(jvmExceptions[i]);
         
      return typeInfoExceptions;
   }
   
   
   public Annotation[][] getParameterAnnotations()
   {
      LoadParameterAnnotationsVisitor visitor = new LoadParameterAnnotationsVisitor();
      getClazz().getReader().readMethod(visitor, AsmClassBytes.STANDARD_FLAGS | ClassReader.INCLUDE_METHOD_PARAMETER_ANNOTATIONS, null, null, getByteCodeIndex());
      return visitor.getAnnotations();
   }
   
   public Annotation[] getAnnotations()
   {
      LoadBehaviourAnnotationsVisitor visitor = new LoadBehaviourAnnotationsVisitor();
      getClazz().getReader().readMethod(visitor, AsmClassBytes.STANDARD_FLAGS | ClassReader.INCLUDE_METHOD_ANNOTATIONS, null, null, getByteCodeIndex());
      return visitor.getAnnotations();
   }
   
   class LoadBehaviourAnnotationsVisitor extends Util.EmptyClassVisitor
   {
      List<Annotation> annotations;
      
      @Override
      public MethodVisitor visitMethod(int access, String name, String desc, String signature, String[] exceptions, int byteCodeIndex)
      {
         if (AsmBehaviourBytes.this.getByteCodeIndex() == byteCodeIndex)
            return new AnnotationReader();
         return null;
      }

      public Annotation[] getAnnotations()
      {
         if (annotations == null)
            return AsmClassBytes.NO_ANNOTATIONS;
         return annotations.toArray(new Annotation[0]);
      }
      
      class AnnotationReader extends Util.EmptyMethodVisitor implements Util.ParentReader
      {
         @Override
         public AnnotationVisitor visitAnnotation(String desc, boolean visible)
         {
            return Util.createAnnotationVisitor(this, AsmBehaviourBytes.this.getClazz().getClassLoader(), desc);
         }

         public void setValueInParent(String name, Object value)
         {
            if (annotations == null)
               annotations = new ArrayList<Annotation>();
            annotations.add((Annotation)value);
         }
      }
   }
   
   class LoadParameterAnnotationsVisitor extends Util.EmptyClassVisitor
   {
      List<Annotation>[] annotations;
      
      @Override
      public MethodVisitor visitMethod(int access, String name, String desc, String signature, String[] exceptions, int byteCodeIndex)
      {
         if (AsmBehaviourBytes.this.getByteCodeIndex() == byteCodeIndex)
            return new AnnotationReader();
         return null;
      }

      Annotation[][] getAnnotations()
      {
         if (annotations == null)
            return null;
         Annotation[][] anns = new Annotation[annotations.length][];
         for (int i = 0 ; i < anns.length ; i++)
            anns[i] = annotations[i] == null ? AsmClassBytes.NO_ANNOTATIONS : annotations[i].toArray(new Annotation[annotations[i].size()]);
         return anns;
      }

      int countParameters()
      {
         //Count the parameters
         String desc = getJvmSignature();
         int params = 0;
         for (int i = 1; i < desc.length(); i++)
         {
            if (desc.charAt(i) == ')')
               break;
            params++;
            while (desc.charAt(i) == '[')
               i++;
            if (desc.charAt(i) == 'L')
            {
               i++;
               while (desc.charAt(i) != ';')
                  i++;
            }
         }
         return params;
      }

      class AnnotationReader extends Util.EmptyMethodVisitor implements Util.ParentReader
      {
         int lastIndex;

         @Override
         public AnnotationVisitor visitParameterAnnotation(int parameter, String desc, boolean visible)
         {
            lastIndex = parameter;
            return Util.createAnnotationVisitor(this, AsmBehaviourBytes.this.getClazz().getClassLoader(), desc);
         }
         
         public void setValueInParent(String name, Object value)
         {
            if (annotations == null)
               annotations = new List[countParameters()];
            if (annotations[lastIndex] == null)
               annotations[lastIndex] = new ArrayList<Annotation>();
            annotations[lastIndex].add((Annotation)value);
         }
      }
   }   
}
