/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.javassist;

import java.util.Arrays;

import javassist.CtBehavior;
import javassist.bytecode.Descriptor;

import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.JBossStringBuilder;


/**
 * SignatureKey.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 105101 $
 */
public class SignatureKey
{
   /** The name */
   String name;
   
   /** The parameter names */
   String[] params;

   /** The cached hashcode */
   private transient int cachedHashCode = Integer.MIN_VALUE;
   
   private static final String[] NO_PARAMS = new String[0];
   
   /**
    * Create a new SignatureKey.
    * 
    * @param name the name
    * @param typeInfos the type infos
    */
   public SignatureKey(String name)
   {
      this.name = name;
      this.params = NO_PARAMS;
   }
   
   /**
    * Create a new SignatureKey.
    * 
    * @param name the name
    * @param typeInfos the type infos
    */
   public SignatureKey(String name, TypeInfo... typeInfos)
   {
      this.name = name;
      if (typeInfos != null && typeInfos.length > 0)
      {
         params = new String[typeInfos.length];
         for (int i = 0; i < typeInfos.length; ++i)
            params[i] = typeInfos[i].getName();
      }
      else
         params = NO_PARAMS;
   }
   
   /**
    * Create a new SignatureKey.
    * 
    * @param name the name
    * @param behavior
    */
   public SignatureKey(String name, CtBehavior behavior)
   {
      this.name = name;
      params = getParameterTypeStringsForSignature(behavior);
   }
   
   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof SignatureKey == false)
         return false;
      
      SignatureKey other = (SignatureKey) obj;
      
      if (name == null && other.name != null)
         return false;
      if (name != null && other.name == null)
         return false;
      if (name != null && name.equals(other.name) == false)
         return false;
      
      if (params == null && other.params == null)
         return true;
      if (params == null && other.params != null)
         return false;
      if (params != null && other.params == null)
         return false;
      
      if (params.length != other.params.length)
         return false;
      
      for (int i = 0; i < params.length; ++i)
      {
         if (params[i].equals(other.params[i]) == false)
            return false;
      }
      return true;
   }
   
   @Override
   public int hashCode()
   {
      if (cachedHashCode == Integer.MIN_VALUE)
      {
         JBossStringBuilder builder = new JBossStringBuilder();
         if (name != null)
            builder.append(name);
         if (params != null)
         {
            for (int i = 0; i < params.length; ++i)
               builder.append(params[i]);
         }
         cachedHashCode = builder.toString().hashCode();
      }
      return cachedHashCode;
   }
   
   @Override
   public String toString()
   {
      if (params == null)
         return name + "[]";
      return name + Arrays.asList(params);
   }
   
   /**
    * Constructs the parameter strings for a behaviour's signature in the same way
    * as TypeInfo does
    * 
    * @param behaviour
    * @return the parameter strings
    */
   private String[] getParameterTypeStringsForSignature(CtBehavior behavior)
   {
      String desc = behavior.getSignature();
      String[] args = new String[Descriptor.numOfParameters(desc)];

      int n = 0;
      int i = 1;
      do
      {
         i = getParameterTypeString(desc, i, args, n++);
      }
      while (i > 0 && n < args.length);
      return args;
   }

   /**
    * Put the next argument into the parameter string array 
    * 
    * @param desc the beahviour's descriptor
    * @param i the current index in the descriptor
    * @param args the parameter string array
    * @param n the current index of the arguments array
    */
   private int getParameterTypeString(String desc, int i, String[] args, int n)
   {
      int i2;
      String name;

      int arrayDim = 0;
      char c = desc.charAt(i);
      if (c == ')')
         return ++i;
      while (c == '[')
      {
         ++arrayDim;
         c = desc.charAt(++i);
      }

      boolean object = false;
      if (c == 'L')
      {
         i2 = desc.indexOf(';', ++i);
         name = desc.substring(i, i2++).replace('/', '.');
         object = true;
      }
      else
      {
         String type = arrayDim == 0 ? toPrimitiveClass(c) : String.valueOf(c);

         i2 = i + 1;
         if (arrayDim == 0)
         {
            args[n] = type;
            return i2; // neither an array type or a class type
         }
         else
            name = type;
      }

      if (arrayDim > 0)
      {
         StringBuilder sb = new StringBuilder();
         while (arrayDim-- > 0)
            sb.append("[");

         if (object)
            sb.append("L");
         sb.append(name);
         if (object)
            sb.append(";");

         name = sb.toString();
      }

      args[n] = name;
      return i2;
   }

   /**
    * Create a primitive entry
    * 
    * @param the character for the primitive
    * @return the primitive real name
    */
   private String toPrimitiveClass(char c)
   {
      String type = null;
      switch (c)
      {
         case 'Z' :
            type = "boolean";
            break;
         case 'C' :
            type = "char";
            break;
         case 'B' :
            type = "byte";
            break;
         case 'S' :
            type = "short";
            break;
         case 'I' :
            type = "int";
            break;
         case 'J' :
            type = "long";
            break;
         case 'F' :
            type = "float";
            break;
         case 'D' :
            type = "double";
            break;
         case 'V' :
            type = "void";
            break;
         default :
            throw new IllegalArgumentException("Unknown primitive type " + c);
      }

      return type;
   }

   public String[] getParams()
   {
      return params;
   }
   
   public boolean isDouble(int i)
   {
      return params[i].equals("double");
   }
   
   public boolean isLong(int i)
   {
      return params[i].equals("long");
   }
}
