/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist.defaultmember;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;

import org.jboss.reflect.plugins.javassist.JavassistConstructor;
import org.jboss.reflect.plugins.javassist.JavassistConstructorInfo;
import org.jboss.reflect.plugins.javassist.JavassistField;
import org.jboss.reflect.plugins.javassist.JavassistFieldInfo;
import org.jboss.reflect.plugins.javassist.JavassistMethod;
import org.jboss.reflect.plugins.javassist.JavassistMethodInfo;
import org.jboss.reflect.spi.TypeInfo;

/**
 * <p>Class to create implementations of the {@link JavassistMethod}, {@link JavassistConstructor} 
 * and {@link JavassistField} interfaces.</p>
 * <p>This implementation simply creates implementations that use reflection</p>
 * 
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class DefaultMemberFactory
{
   private static final Class<?>[] NO_PARAMS = new Class[0];
   
   /**
    * Creates a new {@link JavassistMethod} implementation for a given method
    * 
    * @param methodInfo the CtMethod for which we want to create a JavassistMethod implementation
    * @return the implementing class
    * @throws RuntimeException if an error ocurred
    */
   public static JavassistMethod createJavassistMethod(JavassistMethodInfo methodInfo)
   {
      if (methodInfo == null)
         throw new IllegalArgumentException("Null method");
      
      try
      {
         //Ok to load the class here since this is only called by MethodInfo.invoke() in which
         //case the class must be loaded
         Class<?> clazz = methodInfo.getDeclaringClass().getType();
         Class<?>[] params = getParams(methodInfo.getParameterTypes());
         Method method = clazz.getDeclaredMethod(methodInfo.getName(), params); 
         
         return new DefaultJavassistMethod(method);
      }
      catch(RuntimeException e)
      {
         throw e;
      }
      catch(Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   /**
    * Creates a new {@link JavassistConstructor} implementation for a given constructor
    * 
    * @param constructorInfo the CtConstructor for which we want to create a JavassistConstructor implementation
    * @return the implementing class
    * @throws RuntimeException if an error ocurred
    */
   public static JavassistConstructor createJavassistConstructor(JavassistConstructorInfo constructorInfo)
   {
      if (constructorInfo == null)
         throw new IllegalArgumentException("Null constructor");

      try
      {
         //Ok to load the class here since this is only called by ConstructorInfo.newInstance() in which
         //case the class must be loaded
         Class<?> clazz = constructorInfo.getDeclaringClass().getType();
         Class<?>[] params = getParams(constructorInfo.getParameterTypes());
         Constructor<?> ctor = clazz.getDeclaredConstructor(params); 
         return new DefaultJavassistConstructor(ctor);
      }
      catch(RuntimeException e)
      {
         throw e;
      }
      catch(Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   /**
    * Creates a new {@link JavassistField} implementation for a given field
    * 
    * @param fieldInfo the CtField for which we want to create a JavassistField implementation
    * @return the implementing class
    * @throws RuntimeException if an error ocurred
    */
   public static JavassistField createJavassistField(JavassistFieldInfo fieldInfo)
   {
      if (fieldInfo == null)
         throw new IllegalArgumentException("Null field");

      try
      {
         //Ok to load the class here since this is only called by FieldInfo.get()/set() in which
         //case the class must be loaded
         Class<?> clazz = fieldInfo.getDeclaringClass().getType();
         Field field = clazz.getDeclaredField(fieldInfo.getName()); 
         return new DefaultJavassistField(field);
      }
      catch(RuntimeException e)
      {
         throw e;
      }
      catch(Exception e)
      {
         throw new RuntimeException(e);
      }
   }
   
   private static Class<?>[] getParams(TypeInfo[] typeParams)
   {
      if (typeParams == null || typeParams.length == 0)
         return NO_PARAMS;
      Class<?>[] params = new Class[typeParams.length];
      for (int i = 0 ; i < params.length ; i++)
         params[i] = typeParams[i].getType();
      return params;
   }

}
