/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.javassist;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.CtField;
import javassist.CtMethod;
import javassist.NotFoundException;
import javassist.bytecode.Descriptor;
import javassist.scopedpool.ScopedClassPoolRepository;
import javassist.scopedpool.ScopedClassPoolRepositoryImpl;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class JavassistUtil
{

   /** @todo Per classloader */
   static final ScopedClassPoolRepository repository = ScopedClassPoolRepositoryImpl.getInstance();

   /**
    * Get the javassist class
    * 
    * @param clazz the class
    * @return the javassist method
    */
   public static CtClass getCtClass(Class clazz)
   {
      try
      {
         ClassPool pool = repository.findClassPool(clazz.getClassLoader());
         return pool.get(clazz.getName());
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException("Unable to load CtClass for " + clazz, e);
      }
   }

   /**
    * Get the javassist method
    * 
    * @param method the method
    * @return the javassist method
    */
   public static CtMethod getCtMethod(Method method)
   {
      CtClass clazz = getCtClass(method.getDeclaringClass());
      
      Class[] parameters = method.getParameterTypes();
      CtClass[] params = new CtClass[parameters.length];
      for (int i = 0; i < parameters.length; ++i)
         params[i] = getCtClass(parameters[i]);
   
      CtClass returnType = getCtClass(method.getReturnType());
      String descriptor = Descriptor.ofMethod(returnType, params);
      
      try
      {
         return clazz.getMethod(method.getName(), descriptor);
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException("Unable to find method " + method + " descriptor=" +  descriptor, e);
      }
   }

   /**
    * Get the javassist constructor
    * 
    * @param constructor the constructor
    * @return the javassist constructor
    */
   public static CtConstructor getCtConstructor(Constructor constructor)
   {
      CtClass clazz = getCtClass(constructor.getDeclaringClass());
      
      Class[] parameters = constructor.getParameterTypes();
      CtClass[] params = new CtClass[parameters.length];
      for (int i = 0; i < parameters.length; ++i)
         params[i] = getCtClass(parameters[i]);
   
      String descriptor = Descriptor.ofConstructor(params);
      
      try
      {
         return clazz.getConstructor(descriptor);
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException("Unable to find constructor descriptor=" +  descriptor, e);
      }
   }

   /**
    * Get the javassist field
    * 
    * @param field the field
    * @return the javassist field
    */
   public static CtField getCtField(Field field)
   {
      CtClass clazz = getCtClass(field.getDeclaringClass());
      
      try
      {
         return clazz.getField(field.getName());
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException("Unable to find field " + field, e);
      }
   }

}

