/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.lang;

/**
 * Contains methods added to java.lang.Character in 1.5.
 *
 * Strings in 1.4 don't know about surrogate pairs and the difference between
 * code points and chars, so for now just treat characters as if they were code points.
 */
public final class CharacterRedirects
{

   public static boolean isHighSurrogate(char ch) {
      return false;
   }

   public static boolean isLowSurrogate(char ch) {
      return false;
   }

   public static boolean isSurrogatePair(char high, char low) {
      return false;
   }

   public static int charCount(int codePoint) {
      return 1;
   }

   /**
    * Get the code point for the given high+low surrogate.  Since
    * java 1.4 doesn't support surrogates, the result of this method
    * is always undefined.  Just return '?'.
    *
    * @param high the high surrogate
    * @param low the low surrogate
    * @return the code point
    */
   public static int toCodePoint(char high, char low) {
      return '?';
   }

   public static int codePointAt(CharSequence seq, int index) {
      return seq.charAt(index);
   }

   public static int codePointAt(char[] a, int index) {
      return a[index];
   }

   public static int codePointAt(char[] a, int index, int limit) {
      if (index > limit) {
         throw new IndexOutOfBoundsException("index is greater than limit in Character.codePointAt()");
      }
      return a[index];
   }

   public static int codePointBefore(CharSequence seq, int index) {
      return seq.charAt(index - 1);
   }

   public static int codePointBefore(char[] a, int index) {
      return a[index - 1];
   }

   public static int codePointBefore(char[] a, int index, int start) {
      return a[index + start - 1];
   }

   public static int toChars(int codePoint, char[] dst, int dstIndex) {
      dst[dstIndex] = (char) codePoint;
      return 1;
   }

   public static char[] toChars(int codePoint) {
      return new char[] { (char)codePoint };
   }

   public static int codePointCount(CharSequence seq, int beginIndex, int endIndex) {
      if (beginIndex < 0 || endIndex > seq.length() || beginIndex > endIndex) {
         throw new IndexOutOfBoundsException("Arguments provided to codePointCount are out of range");
      }
      return endIndex - beginIndex;
   }

   public static int codePointCount(char[] a, int offset, int count) {
      if (offset < 0 || count < 0 || offset + count > a.length) {
         throw new IndexOutOfBoundsException("Arguments provided to codePointCount are out of range");
      }
      return count;
   }

   public static int offsetByCodePoints(CharSequence seq, int index, int codePointOffset) {
      final int length = seq.length();
      if (index > length || codePointOffset + index > length || codePointOffset + index < 0) {
         throw new IndexOutOfBoundsException("Arguments provided to offsetByCodePoints are out of range");
      }
      return index + codePointOffset;
   }

   public static int offsetByCodePoints(char[] a, int start, int count, int index, int codePointOffset) {
      if (index + start > count || codePointOffset + index + start > count || codePointOffset + index + start < 0) {
         throw new IndexOutOfBoundsException("Arguments provided to offsetByCodePoints are out of range");
      }
      return start + index + codePointOffset;
   }

   public static boolean isLowerCase(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isLowerCase((char)codePoint);
   }

   public static boolean isUpperCase(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isUpperCase((char)codePoint);
   }

   public static boolean isTitleCase(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isTitleCase((char)codePoint);
   }

   public static boolean isDigit(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isDigit((char)codePoint);
   }

   public static boolean isDefined(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isDefined((char)codePoint);
   }

   public static boolean isLetter(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isLetter((char)codePoint);
   }

   public static boolean isLetterOrDigit(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isLetterOrDigit((char)codePoint);
   }

   public static boolean isJavaIdentifierStart(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isJavaIdentifierStart((char)codePoint);
   }

   public static boolean isJavaIdentifierPart(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isJavaIdentifierPart((char)codePoint);
   }

   public static boolean isUnicodeIdentifierStart(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isUnicodeIdentifierStart((char)codePoint);
   }

   public static boolean isUnicodeIdentifierPart(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isUnicodeIdentifierPart((char)codePoint);
   }

   public static boolean isIdentifierIgnorable(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isIdentifierIgnorable((char)codePoint);
   }

   public static int toLowerCase(int codePoint) {
      return codePoint <= Character.MAX_VALUE ? Character.toLowerCase((char)codePoint) : codePoint;
   }

   public static int toUpperCase(int codePoint) {
      return codePoint <= Character.MAX_VALUE ? Character.toUpperCase((char)codePoint) : codePoint;
   }

   public static int toTitleCase(int codePoint) {
      return codePoint <= Character.MAX_VALUE ? Character.toTitleCase((char)codePoint) : codePoint;
   }

   public static int digit(int codePoint, int radix) {
      return codePoint <= Character.MAX_VALUE ? Character.digit((char)codePoint, radix) : -1;
   }

   public static int getNumericValue(int codePoint) {
      return codePoint <= Character.MAX_VALUE ? Character.getNumericValue((char)codePoint) : -1;
   }

   public static boolean isSpaceChar(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isSpaceChar((char)codePoint);
   }

   public static boolean isWhitespace(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isWhitespace((char)codePoint);
   }

   public static boolean isISOControl(int codePoint) {
      return codePoint <= Character.MAX_VALUE && Character.isISOControl((char)codePoint);
   }

   public static int getType(int codePoint) {
      return codePoint <= Character.MAX_VALUE ? Character.getType((char)codePoint) : Character.UNASSIGNED;
   }

   public static byte getDirectionality(int codePoint) {
      return codePoint <= Character.MAX_VALUE ? Character.getDirectionality((char)codePoint) : Character.DIRECTIONALITY_UNDEFINED;
   }

   public static boolean isMirrored(int codePoint) {
      return codePoint <= Character.MAX_VALUE ? Character.isMirrored((char)codePoint) : false;
   }

   public static char reverseBytes(char ch) {
      return (char)((ch << 8) | (ch >>> 8));
   }
}
