/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.lang;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.ArrayList;


/**
 * Implementations of java.lang.Class methods added in jdk5
 *
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author Scott.Stark@jboss.org
 * @version $Revision: 207 $
 */
public class ClassRedirects
{
   /** The enum modifier */
   // private static final int ENUM = 0x00004000;
   /**
    * The synthetic modifier
    */
   private static final int SYNTHETIC = 0x00001000;


   /**
    * Whether the class is an annotation
    *
    * @param clazz the class
    * @return true when it is an annotation
    */
   public static boolean isAnnotation(Class clazz)
   {
      Class[] interfaces = clazz.getInterfaces();

      for (int i = 0 ; i < interfaces.length ; i++)
      {
         if (interfaces[i].getName().equals("java.lang.Annotation"))
         {
            return true;
         }
         if (interfaces[i].getName().equals(Annotation.class.getName()))
         {
            return true;
         }
         if (isAnnotation(interfaces[i]))
         {
            return true;
         }
      }
      return false;
   }

   /**
    * Whether the class is an enum
    *
    * @param clazz the class
    * @return true when it is an enum
    */
   public static boolean isEnum(Class clazz)
   {
      // @todo modifiers
      //if ((ENUM & clazz.getModifiers()) == 0)
      //   return false;
      Class superClass = clazz.getSuperclass();
      if (superClass == null)
      {
         return false;
      }
      if (EnumImpl.class == clazz.getSuperclass())
      {
         return true;
      }
      return superClass.getName().equals("java.lang.Enum");
   }

   /**
    * Get the enumeration constants
    *
    * @param <T> the enum type
    * @param enumType the enum type
    * @return the constants
    */
   public static <T extends EnumImpl<T>> T[] getEnumConstants(Class<T> enumType)
   {
      ArrayList<T> list = EnumImpl.getEnumConstants(enumType);
      Object[] array = (Object[]) Array.newInstance(enumType, list.size());
      return (T[]) list.toArray(array);
   }

   /**
    * Cast an object
    *
    * @param <T>   the expected type
    * @param clazz the expected class
    * @param o     the object
    * @return the cast object
    */
   public static <T> T cast(Class<T> clazz, Object o)
   {
      if (o != null && clazz.isInstance(o) == false)
      {
         throw new ClassCastException("Expected " + clazz.getName() + " got " + o.getClass().getName());
      }
      return (T) o;
   }

   /**
    * @param clazz
    * @param thisClazz
    * @return thisClazz object, cast to represent a subclass of
    *         the specified class object.
    * @throws ClassCastException if thisClazz object does not
    *  represent a subclass of the specified class (here "subclass" includes
    *  the class itself).
    */
   public static <T> Class<? extends T> asSubclass(Class<T> clazz, Class thisClazz)
   {
      if (clazz.isAssignableFrom(thisClazz))
      {
         return (Class<? extends T>) thisClazz;
      }
      else
      {
         throw new ClassCastException("Expected " + clazz.getName() + " got " + thisClazz.getName());
      }
   }

   /**
    * Replace a character sequence
    *
    * @param string the string
    * @param target the string to replace
    * @param replacement the replacement text
    * @return the result
    * @deprecated As of release 1.1.0 replaced by {@link StringRedirects#replace(String, CharSequence, CharSequence) StringRedirects.replace}
    */
   public static String replace(String string, CharSequence target, CharSequence replacement)
   {
      return StringRedirects.replace(string, target, replacement);
   }

   /**
    * Whether the string contains the character sequence
    *
    * @param string the string
    * @param s      the character sequence
    * @return true when it contains the sequence
    * @deprecated  As of release 1.1.0 replaced by {@link StringRedirects#contains(String, CharSequence) StringRedirects.contains }
    */
   public static boolean contains(String string, CharSequence s)
   {
      return StringRedirects.contains(string, s);
   }

   /**
    * Quote the replacement string
    *
    * @param string the quoted string
    * @deprecated As of release 1.1.0 replaced by {@link StringRedirects#quoteReplacement(String) StringRedirects.quoteReplacement }
    */
   public static String quoteReplacement(String string)
   {
      return StringRedirects.quoteReplacement(string);
   }

   /**
    * Implementation of the jdk5 Class.getSimpleName() method.
    * Returns the simple name of the underlying class as given in the
    * source code. Returns an empty string if the underlying class is
    * anonymous.
    * <p/>
    * <p>The simple name of an array is the simple name of the
    * component type with "[]" appended.  In particular the simple
    * name of an array whose component type is anonymous is "[]".
    *
    * @param clazz the Class instance
    * @return the simple name of the underlying class
    */
   public static String getSimpleName(Class clazz)
   {
      boolean isArray = clazz.isArray();
      if (isArray)
      {
         String arrayName = getSimpleName(clazz.getComponentType())+"[]";
         return arrayName;
      }
      // Remove the package name
      String name = clazz.getName();
      int dot = name.lastIndexOf('.');
      if( dot >= 0 )
      {
         name = name.substring(dot+1);
         // Check for inner class
         int dollar = name.lastIndexOf('$');
         if( dollar >= 0 )
         {
            name = name.substring(dollar+1);
            // Check for anonymous $N classes (N=0-9)
            int index = 0;
            for(; index < name.length() && Character.isDigit(name.charAt(index)); index ++)
               ;
            if( index > 0 )
               name = name.substring(index);
         }
      }
      return name;
   }

   /**
    * Returns the canonical name of the the underlying class as
    * defined by the Java Language Specification.  Returns null if
    * the underlying class does not have a canonical name (i.e., if
    * it is a local or anonymous class or an array whose component
    * type does not have a canonical name).
    *
    * The difference between a fully qualified name and a canonical name can be
    * seen in examples such as:
    package p;
    class O1 { class I{}}
    class O2 extends O1{};
    * In this example both p.O1.I and p.O2.I are fully qualified names that
    * denote the same class, but only p.O1.I is its canonical name.
    *
    * @param clazz the Class instance
    * @return the canonical name of the underlying class if it exists, and
    *         null otherwise.
    */
   public static String getCanonicalName(Class clazz)
   {
      if (clazz.isArray())
      {
         String name = clazz.getComponentType().getCanonicalName();
         if( name != null )
            name += "[]";
         return name;
      }
      String name = null;
      if( isAnonymousClass(clazz) == false )
      {
         // this is not correct for nested classes?
         name = clazz.getName();
         name = name.replace('$', '.');
      }
      return name;
   }

   /**
    * Returns true if and only if the underlying class
    * is an anonymous class.
    *
    * @param clazz the Class instance
    * @return true if clazz is an anonymous class.
    * @since 1.5
    */
   public static boolean isAnonymousClass(Class clazz)
   {
      String name = getSimpleName(clazz);
      return name.length() == 0;
   }

   /**
    * @param clazz
    * @return true if (clazz.getModifiers() & SYNTHETIC) != 0
    * @todo
    */
   public static boolean isSynthetic(Class clazz)
   {
      throw new UnsupportedOperationException("isSynthetic is a TODO");
   }

   /**
    * @param clazz the Class instance
    * @return true if and only if this class is a local class.
    * @todo Returns true if and only if the underlying class
    * is a local class.
    * @since 1.5
    */
   public static boolean isLocalClass(Class clazz)
   {
      throw new UnsupportedOperationException("isLocalClass is a TODO");
   }

   /**
    * @param clazz the Class instance
    * @return true if and only if this class is a member class.
    * @todo Returns true if and only if the underlying class
    * is a member class.
    * @since 1.5
    */
   public static boolean isMemberClass(Class clazz)
   {
      throw new UnsupportedOperationException("isMemberClass is a TODO");
   }

   /**
    * @param clazz
    * @return an array of TypeVariable objects that represent
    *         the type variables declared by this generic declaration
    * @todo
    */
   /*public static <T> TypeVariable<Class<T>>[] getTypeParameters(Class clazz)
   {
      throw new UnsupportedOperationException("getTypeParameters is a TODO");
   }*/

   /**
    * @param clazz
    * @return the superclass of the class represented by this object
    * @todo
    */
   public static Object getGenericSuperclass(Class clazz)
   {
      return GenericsHelper.getGenericSuperclass(clazz);
   }

   /**
    * @param clazz
    * @return an array of interfaces implemented by this class
    * @todo
    */
   public static Object[] getGenericInterfaces(Class clazz)
   {
      return GenericsHelper.getGenericInterfaces(clazz);
   }

   /**
    * This method cannot be implemented as it relies on native method info.
    *
    * @param clazz
    * @return null
    */
   public static Method getEnclosingMethod(Class clazz)
   {
      throw new UnsupportedOperationException("getEnclosingMethod cannot be implemented");
   }

   /**
    * This method cannot be implemented as it relies on native method info.
    *
    * @param clazz
    * @return null
    */
   public static Constructor<?> getEnclosingConstructor(Class clazz)
   {
      throw new UnsupportedOperationException("getEnclosingConstructor cannot be implemented");
   }

   /**
    * This method cannot be implemented as it relies on native method info.
    *
    * @param clazz
    * @return null
    */
   public static Class<?> getEnclosingClass(Class clazz)
   {
      throw new UnsupportedOperationException("getEnclosingClass cannot be implemented");
   }

   /**
    * Static helper class
    */
   private ClassRedirects()
   {
   }
}
