/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.lang;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.WeakHashMap;

/**
 * Mapping of the java.lang.Enum.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author Scott.Stark@jboss.org
 * @version $Revision: 200 $
 */
public abstract class EnumImpl<E extends EnumImpl<E>> implements Serializable, Comparable<E>
{
   private static final long serialVersionUID = 3760558672530976823L;
   private static WeakHashMap<Class, ArrayList> enumConstants = new WeakHashMap<Class, ArrayList>();

   private final String name;
   private final int ordinal;

   /**
    * Obtain the Enum of enumType for the given name.
    * 
    * @param enumType - the class for the enum type
    * @param name - the string representation of the enum type
    * @return the T enum type if found, null otherwise
    */
   public static <T extends EnumImpl<T>> T valueOf(Class<T> enumType, String name)
   {
      ArrayList<T> enums = getEnumConstants(enumType);
      
      T match = null;
      for(T t : enums)
      {
         if( t.name().equals(name) )
         {
            match = t;
            break;
         }
      }
      return match;
   }
   /**
    * Obtain the array of Enums for enumType.
    * 
    * @param enumType - the class for the enum type
    * @return the T[] enum types if enumType is a registered type, null otherwise
    */
   public static <T extends EnumImpl<T>> T[] values(Class<T> enumType)
   {
      ArrayList<T> enums = getEnumConstants(enumType);
      T[] values = null;
      if( enums != null )
      {
         values = (T[]) enums.toArray();
      }
      return values;
   }

   static <T extends EnumImpl<T>> ArrayList<T> getEnumConstants(Class<T> enumType)
   {
      ArrayList<T> enums =  enumConstants.get(enumType);
      
      if (enums == null)
      {
         enums = forceInitialization(enumType);
      }
      
      return enums;
   }
   
   final static Class[] NO_PARAMS = new Class[0];
   final static Object[] NO_VALUES = new Object[0];
   
   static synchronized <T extends EnumImpl<T>> ArrayList<T> forceInitialization(final Class<T> enumType)
   {
      try
      {
         //Force initialisation of the woven enum
         Method valuesMethod = enumType.getMethod("values", NO_PARAMS);
         Object values = valuesMethod.invoke(null,NO_VALUES);
         ArrayList<T> enums = new ArrayList<T>();
         for (T elem : (T[])values) {
            enums.add(elem);
         }
         
         if (enums == null)
         {
            enums = new ArrayList<T>();
            enumConstants.put(enumType, enums);
         }
         return enums;
      }
      catch (Exception e)
      {
         // AutoGenerated
         throw new RuntimeException(e);
      }
   }
   /**
    * Create an enum type. This registers the enum with the class enumConstants
    * to support the valueOf, values methods.
    * 
    * @see org.jboss.weaver.retro.WeaverRetroJdk14#rewriteEnum(ClassLoader, CompilerClassInfo)
    * 
    * @param name - the string representation of the enum
    * @param ordinal - the int representation of the enum
    */
   protected EnumImpl(String name, int ordinal)
   {
      this.name = name;
      this.ordinal = ordinal;
      synchronized(enumConstants)
      {
         ArrayList<EnumImpl> enums = enumConstants.get(getClass());
         if( enums == null )
         {
            enums = new ArrayList<EnumImpl>();
            enumConstants.put(getClass(), enums);
         }
         enums.add(this);
      }
   }

   public final String name()
   {
      return name;
   }
   
   public final int ordinal()
   {
      return ordinal;
   }
   
   public String toString()
   {
      return name;
   }
   
   public final boolean equals(Object other)
   {
      return this == other;
   }
   
   public int hashCode()
   {
      return ordinal;
   }
   
   public final int compareTo(E o)
   {
      return o.ordinal - ordinal;
   }
   
   protected final Object clone() throws CloneNotSupportedException
   {
      throw new CloneNotSupportedException("An Enum is not cloneable!");
   }
   
   public final Class<E> getDeclaringClass()
   {
      Class thisClass = getClass();
      Class superClass = thisClass.getSuperclass();
      if (superClass == EnumImpl.class)
         return thisClass;
      else
         return superClass;
   }
   
}
