/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.lang;

import java.util.regex.Pattern;
import java.util.regex.Matcher;

/**
 * Contains methods added to java.lang.String in 1.5.
 *
 * Strings in 1.4 don't know about surrogate pairs and the difference between
 * code points and chars, so for now just treat characters as if they were code points.
 */
public class StringRedirects
{
   /**
    * Replace a character sequence
    *
    * @param string the string
    * @param target the string to replace
    * @param replacement the replacement text
    * @return the result
    */
   public static String replace(String string, CharSequence target, CharSequence replacement)
   {
      Pattern pattern = Pattern.compile(target.toString(), Pattern.LITERAL);
      Matcher matcher = pattern.matcher(string);
      String quotedReplacement = quoteReplacement(replacement.toString());
      return matcher.replaceAll(quotedReplacement);
   }

   /**
    * Whether the string contains the character sequence
    *
    * @param string the string
    * @param s      the character sequence
    * @return true when it contains the sequence
    */
   public static boolean contains(String string, CharSequence s)
   {
      return string.indexOf(s.toString()) != -1;
   }

   /**
    * Quote the replacement string
    *
    * @param string the quoted string
    */
   public static String quoteReplacement(String string)
   {
      StringBuilder builder = new StringBuilder();
      for (char c : string.toCharArray())
      {
         if (c == '\\')
         {
            builder.append("\\\\");
         }
         else if (c == '$')
         {
            builder.append("\\$");
         }
         else
         {
            builder.append(c);
         }
      }
      return builder.toString();
   }

   /**
    * Returns the character (Unicode code point) at the specified index.  Since
    * java 1.4 doesn't know about code points, just returns the character value.
    *
    * @param string the string to examine
    * @param index the code point index
    * @return the code point
    * @throws IndexOutOfBoundsException if the index argument is less than zero or
    *  beyond the end of the string
    */
   public static int codePointAt(String string, int index)
   {
      return string.charAt(index);
   }

   /**
    * Returns the character (Unicode code point) just before the specified index.  Since
    * java 1.4 doesn't know about code points, just returns the character value.
    *
    * @param string the string to examine
    * @param index the code point index
    * @return the code point
    * @throws IndexOutOfBoundsException if the index argument is less than one or
    *  beyond the end of the string
    */
   public static int codePointBefore(String string, int index)
   {
      return string.charAt(index - 1);
   }

   /**
    * Returns the number of code points between in the specified range.  Since
    * java 1.4 doesn't know about code points, just returns character count.
    *
    * @param string the string to examine
    * @param beginIndex the code point index to start at
    * @param endIndex the code point index to end at
    * @return the code point difference
    * @throws IndexOutOfBoundsException if the beginIndex is negative, or endIndex
    *   is larger than the length of this String, or beginIndex is larger than endIndex.
    */
   public static int codePointCount(String string, int beginIndex, int endIndex)
   {
      if (beginIndex < 0 || endIndex > string.length() || beginIndex > endIndex) {
         throw new IndexOutOfBoundsException("Arguments provided to codePointCount are out of range");
      }
      return endIndex - beginIndex;
   }

   /**
    * Returns the index within this String that is offset from the given index by codePointOffset code points.
    * Since java 1.4 doesn't know about code points, just returns character index.
    *
    * @param string the string to examine
    * @param index the index to be offset
    * @param codePointOffset the offset in code points
    * @return the index within the string
    * @throws IndexOutOfBoundsException if index is negative or larger then the length of this String,
    *    or if codePointOffset is positive and the substring starting with index has fewer than
    *    codePointOffset code points, or if codePointOffset is negative and the substring before
    *    index has fewer than the absolute value of codePointOffset code points.
    */
   public static int offsetByCodePoints(String string, int index, int codePointOffset)
   {
      final int length = string.length();
      if (index < 0 || index > length || codePointOffset + index > length || codePointOffset + index < 0) {
         throw new IndexOutOfBoundsException("Arguments provided to offsetByCodePoints are out of range");
      }
      return index + codePointOffset;
   }

   // StringBuffer redirects

   public static StringBuffer appendCodePoint(StringBuffer buffer, int codePoint) {
      return buffer.append((char)codePoint);
   }

   public static int codePointAt(StringBuffer buffer, int index) {
      return buffer.charAt(index);
   }

   public static int codePointBefore(StringBuffer buffer, int index) {
      return buffer.charAt(index - 1);
   }

   public static int codePointCount(StringBuffer buffer, int beginIndex, int endIndex) {
      return endIndex - beginIndex;
   }

   public static int offsetByCodePoints(StringBuffer buffer, int index, int codePointOffset) {
      return index + codePointOffset;
   }

   // StringBuilder redirects

   public static JBossStringBuilder appendCodePoint(JBossStringBuilder builder, int codePoint) {
      return builder.append((char)codePoint);
   }

   public static int codePointAt(JBossStringBuilder builder, int index) {
      return builder.charAt(index);
   }

   public static int codePointBefore(JBossStringBuilder builder, int index) {
      return builder.charAt(index - 1);
   }

   public static int codePointCount(JBossStringBuilder builder, int beginIndex, int endIndex) {
      return endIndex - beginIndex;
   }

   public static int offsetByCodePoints(JBossStringBuilder builder, int index, int codePointOffset) {
      return index + codePointOffset;
   }
}
