/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.util;

import java.io.Serializable;
import java.util.AbstractSet;
import java.util.Collection;
import java.util.Iterator;
import java.util.TreeSet;

/**
 * EnumSet.
 *
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 200 $
 */
public class EnumSet<E extends Enum> extends AbstractSet<E> implements Cloneable, Serializable
{
   /** The serialVersionUID @todo fix the serialization */
   private static final long serialVersionUID = -3136546369353376452L;

   // @todo bitset as described in the javadoc (using TreeSet for iterator ordering)
   private TreeSet<E> values;
   
   private final Class<E> enumType;
   
   private final Enum[] constants;  
   
   private EnumSet(Class<E> enumType)
   {
      if (enumType.isEnum() == false)
         throw new IllegalArgumentException("Not an enum " + enumType);
      this.enumType = enumType;
      values = new TreeSet();
      constants = enumType.getEnumConstants();
   }
   
   public static <E extends Enum<E>> EnumSet<E> noneOf(Class<E> enumType)
   {
      return new EnumSet(enumType);
   }

   public static <E extends Enum<E>> EnumSet<E> allOf(Class<E> enumType)
   {
      EnumSet result = new EnumSet(enumType);
      for (Enum e : result.constants)
         result.add(e);
      return result;
   }

   public static <E extends Enum<E>> EnumSet<E> copyOf(EnumSet<E> s)
   {
      return s.clone();
   }

   public static <E extends Enum<E>> EnumSet<E> copyOf(Collection<E> c)
   {
      if (c instanceof EnumSet)
         return ((EnumSet) c).clone();
      else
      {
         if (c.isEmpty())
            throw new IllegalArgumentException("Empty collection");
         Iterator<E> iter = c.iterator();
         E first = iter.next();
         EnumSet result = of(first);
         while (iter.hasNext())
            result.add(iter.next());
         return result;
      }
   }

   public static <E extends Enum<E>> EnumSet<E> complementOf(EnumSet<E> s)
   {
      EnumSet result = allOf(s.enumType);
      result.removeAll(s);
      return result;
   }

   public static <E extends Enum<E>> EnumSet<E> of(E e)
   {
      EnumSet result = new EnumSet(e.getDeclaringClass());
      result.add(e);
      return result;
   }

   public static <E extends Enum<E>> EnumSet<E> of(E e1, E e2)
   {
      EnumSet result = new EnumSet(e1.getDeclaringClass());
      result.add(e1);
      result.add(e2);
      return result;
   }

   public static <E extends Enum<E>> EnumSet<E> of(E e1, E e2, E e3)
   {
      EnumSet result = new EnumSet(e1.getDeclaringClass());
      result.add(e1);
      result.add(e2);
      result.add(e3);
      return result;
   }

   public static <E extends Enum<E>> EnumSet<E> of(E e1, E e2, E e3, E e4)
   {
      EnumSet result = new EnumSet(e1.getDeclaringClass());
      result.add(e1);
      result.add(e2);
      result.add(e3);
      result.add(e4);
      return result;
   }

   public static <E extends Enum<E>> EnumSet<E> of(E e1, E e2, E e3, E e4, E e5)
   {
      EnumSet result = new EnumSet(e1.getDeclaringClass());
      result.add(e1);
      result.add(e2);
      result.add(e3);
      result.add(e4);
      result.add(e5);
      return result;
   }

   public static <E extends Enum<E>> EnumSet<E> of(E first, E... rest)
   {
      EnumSet result = EnumSet.of(first);
      for (E e : rest)
         result.add(e);
      return result;
   }

   public static <E extends Enum<E>> EnumSet<E> range(E from, E to)
   {
      EnumSet result = new EnumSet(from.getDeclaringClass());
      for (int i = from.ordinal(); i <= to.ordinal(); ++i)
         result.add(result.constants[i]);
      return result;
   }

   public EnumSet<E> clone()
   {
      EnumSet result;
      try
      {
         result = (EnumSet<E>) super.clone();
      }
      catch (CloneNotSupportedException e)
      {
         throw new Error(e);
      }
      result.values = (TreeSet<E>) values.clone();
      return result;
   }

   public Iterator<E> iterator()
   {
      return values.iterator();
   }

   public int size()
   {
      return values.size();
   }

   public boolean add(E o)
   {
      return values.add(o);
   }
}
