/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.util.concurrent.atomic;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

/**
 * AtomicLongFieldUpdater.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 200 $
 */
public abstract class AtomicLongFieldUpdater<T>
{
   public static <U> AtomicLongFieldUpdater<U> newUpdater(Class<U> tClass, String fieldName)
   {
      return new Implementation<U>(tClass, fieldName);
   }
   
   /**
    * Create a new AtomicLongFieldUpdater.
    */
   protected AtomicLongFieldUpdater()
   {
   }
   
   public long addAndGet(T obj, long delta)
   {
      while (true)
      {
         long expect = get(obj);
         long update = expect + delta;
         if (compareAndSet(obj, expect, update))
            return update;
      }
   }

   public abstract boolean compareAndSet(T obj, long expect, long update);

   public long decrementAndGet(T obj)
   {
      while (true)
      {
         long expect = get(obj);
         long update = expect - 1;
         if (compareAndSet(obj, expect, update))
            return update;
      }
   }

   public abstract long get(T obj);

   public long getAndAdd(T obj, long delta)
   {
      while (true)
      {
         long expect = get(obj);
         long update = expect + delta;
         if (compareAndSet(obj, expect, update))
            return expect;
      }
   }

   public long getAndDecrement(T obj)
   {
      while (true)
      {
         long expect = get(obj);
         long update = expect - 1;
         if (compareAndSet(obj, expect, update))
            return expect;
      }
   }

   public long getAndIncrement(T obj)
   {
      while (true)
      {
         long expect = get(obj);
         long update = expect + 1;
         if (compareAndSet(obj, expect, update))
            return expect;
      }
   }

   public long getAndSet(T obj, long newValue)
   {
      while (true)
      {
         long expect = get(obj);
         if (compareAndSet(obj, expect, newValue))
            return expect;
      }
   }

   public long incrementAndGet(T obj)
   {
      while (true)
      {
         long expect = get(obj);
         long update = expect + 1;
         if (compareAndSet(obj, expect, update))
            return update;
      }
   }

   public abstract void set(T obj, long newValue);

   public abstract boolean weakCompareAndSet(T obj, long expect, long update);
   
   private static class Implementation<T> extends AtomicLongFieldUpdater<T>
   {
      /** The class */
      private Class<T> tClass;
      
      /** The field */
      private String fieldName;
      
      /** The field */
      private Field field;
      
      public Implementation(Class<T> tClass, String fieldName)
      {
         if (tClass == null)
            throw new IllegalArgumentException("Null class");
         if (fieldName == null)
            throw new IllegalArgumentException("Null field name");
         this.tClass = tClass;
         this.fieldName = fieldName;
         try
         {
            field = tClass.getDeclaredField(fieldName);
         }
         catch (Exception e)
         {
            throw new IllegalArgumentException("Cannot find field class=" + tClass.getName() + " field=" + fieldName, e);
         }
         if (Long.TYPE != field.getType())
            throw new IllegalArgumentException("Field is not a long class=" + tClass.getName() + " field=" + fieldName);
         if (Modifier.isVolatile(field.getModifiers()) == false)
            throw new IllegalArgumentException("Field is not volatile class=" + tClass.getName() + " field=" + fieldName);
         field.setAccessible(true);
      }

      public boolean compareAndSet(T obj, long expect, long update)
      {
         try
         {
            synchronized (this)
            {
               long value = field.getLong(obj);
               if (value != expect)
                  return false;
               field.setLong(obj, update);
               return true;
            }
         }
         catch (Exception e)
         {
            throw new RuntimeException("Unexpected error class=" + tClass.getName() + " field=" + fieldName, e);
         }
      }

      public long get(T obj)
      {
         try
         {
            synchronized (this)
            {
               return field.getLong(obj);
            }
         }
         catch (Exception e)
         {
            throw new RuntimeException("Unexpected error class=" + tClass.getName() + " field=" + fieldName, e);
         }
      }

      public void set(T obj, long newValue)
      {
         try
         {
            synchronized (this)
            {
               field.setLong(obj, newValue);
            }
         }
         catch (Exception e)
         {
            throw new RuntimeException("Unexpected error class=" + tClass.getName() + " field=" + fieldName, e);
         }
      }

      public boolean weakCompareAndSet(T obj, long expect, long update)
      {
         return compareAndSet(obj, expect, update);
      }
   }
}
