/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.util.concurrent.atomic;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

/**
 * AtomicReferenceFieldUpdater.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 200 $
 */
public abstract class AtomicReferenceFieldUpdater<T, V>
{
   public static <U, W> AtomicReferenceFieldUpdater<U, W> newUpdater(Class<U> tClass, Class<W> vClass, String fieldName)
   {
      return new Implementation<U, W>(tClass, vClass, fieldName);
   }
   
   /**
    * Create a new AtomicReferenceFieldUpdater.
    */
   protected AtomicReferenceFieldUpdater()
   {
   }

   public abstract boolean compareAndSet(T obj, V expect, V update);

   public abstract V get(T obj);

   public V getAndSet(T obj, V newValue)
   {
      while (true)
      {
         V expect = get(obj);
         if (compareAndSet(obj, expect, newValue))
            return expect;
      }
   }

   public abstract void set(T obj, V newValue);

   public abstract boolean weakCompareAndSet(T obj, V expect, V update);
   
   private static class Implementation<T, V> extends AtomicReferenceFieldUpdater<T, V>
   {
      /** The class */
      private Class<T> tClass;
      
      /** The field */
      private String fieldName;
      
      /** The field */
      private Field field;
      
      public Implementation(Class<T> tClass, Class<V> vClass, String fieldName)
      {
         if (tClass == null)
            throw new IllegalArgumentException("Null type class");
         if (vClass == null)
            throw new IllegalArgumentException("Null value class");
         if (fieldName == null)
            throw new IllegalArgumentException("Null field name");
         this.tClass = tClass;
         this.fieldName = fieldName;
         try
         {
            field = tClass.getDeclaredField(fieldName);
         }
         catch (Exception e)
         {
            throw new IllegalArgumentException("Cannot find field class=" + tClass.getName() + " field=" + fieldName, e);
         }
         if (field.getType().isAssignableFrom(vClass))
            throw new IllegalArgumentException("Field is not a " + vClass.getName() + " class=" + tClass.getName() + " field=" + fieldName);
         if (Modifier.isVolatile(field.getModifiers()) == false)
            throw new IllegalArgumentException("Field is not volatile class=" + tClass.getName() + " field=" + fieldName);
         field.setAccessible(true);
      }

      public boolean compareAndSet(T obj, V expect, V update)
      {
         try
         {
            synchronized (this)
            {
               V value = (V) field.get(obj);
               if (value != expect)
                  return false;
               field.set(obj, update);
               return true;
            }
         }
         catch (Exception e)
         {
            throw new RuntimeException("Unexpected error class=" + tClass.getName() + " field=" + fieldName, e);
         }
      }

      public V get(T obj)
      {
         try
         {
            synchronized (this)
            {
               return (V) field.get(obj);
            }
         }
         catch (Exception e)
         {
            throw new RuntimeException("Unexpected error class=" + tClass.getName() + " field=" + fieldName, e);
         }
      }

      public void set(T obj, V newValue)
      {
         try
         {
            synchronized (this)
            {
               field.set(obj, newValue);
            }
         }
         catch (Exception e)
         {
            throw new RuntimeException("Unexpected error class=" + tClass.getName() + " field=" + fieldName, e);
         }
      }

      public boolean weakCompareAndSet(T obj, V expect, V update)
      {
         return compareAndSet(obj, expect, update);
      }
   }
}
