/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.weaver;

import java.io.IOException;
import java.util.logging.Logger;

/**
 * Takes jar or class files and weaves the bytecode
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 *
 */
public class Main
{
   
   private static Logger log = Logger.getLogger(Main.class.getName());
   
   /**
    * Parses command line arguments and calls setters.
    * @param args
    */
   public static void parseArgs(WeaveRunner weaveRunner, String[] args)
   {
      if (args.length == 0 || args[0].equalsIgnoreCase("-h") || args[0].equalsIgnoreCase("-help"))
      {
         printUsageAndExit();
      }
      String weaverClass = null;
      String classPath = null;
            
      for (int i = 0; i < args.length; ++i)
      {
         if (args[i].equals("-verbose"))
         {
            weaveRunner.setVerbose(true);        
         }
         else if (args[i].equals("-suppress"))
         {
            weaveRunner.setSuppress(true);
         }
         else if (args[i].equals("-outputPath"))
         {
            weaveRunner.setOutputPath(args[++i]);
         }
         else if (args[i].equals("-cp") || args[i].equals("-classpath"))
         {
            if (i + 1 > args.length - 1)
            {
               printUsageAndExit();
            }
            classPath = args[++i];
         }
         else if (args[i].equals("--SOURCEPATH"))
         {
            if (i + 1 > args.length - 1)
            {
               printUsageAndExit();
            }
            weaveRunner.addFilesFromSourcePathFile(args[++i]);
         }
         else if (args[i].equalsIgnoreCase("-weaverClass"))
         {
            if (i + 1 > args.length - 1)
            {
               printUsageAndExit();
            }
            weaverClass = args[++i];
         }
         else
         {
            try
            {
               weaveRunner.addSourcePath(args[i]);
            }
            catch (IOException ioe)
            {
               throw new IllegalArgumentException("Problem accessing source path: " + ioe);
            }
         }
      }
      Weaver weaver = null;
      if (weaverClass == null) {
         weaver = new Weaver();
      } else {
         try {
            weaver = (Weaver)Class.forName(weaverClass).newInstance();
         } catch (Exception e) {
            log.severe("Unable to instantiate weaver class");
            log.severe(e.getMessage());
            System.exit(1);
         }
      }
      weaver.setClasspath(classPath);
      weaver.init();
      weaveRunner.setWeaver(weaver);
   }

   /**
    * Main method can be called from the command line.
    * Usage: java org.jboss.retro.Main [-cp <classpath>] [-classpath <classpath>] [-outputToJar] [-verbose] [-weaver <WeaverClass>] [--SOURCEPATH <source-path-file>] <dir>+
    * @param args command line arguments
    * @throws Exception if anything goes wrong
    */
   public static void main(String[] args)
   {
      long start = System.currentTimeMillis();
      
      WeaveRunner weaveRunner = new WeaveRunner();
      
      parseArgs(weaveRunner, args);
      try
      {
         weaveRunner.weave();
         System.out.println("Build Successful: " + (System.currentTimeMillis() - start) + " ms");
      }
      catch (Exception e)
      {
         log.warning(e.getMessage());
      }
   }

   private static void printUsageAndExit()
   {
      usage();
      System.exit(1);
   }

   public static void usage()
   {
      StringBuffer usage = new StringBuffer();
      usage.append("usage: java org.jboss.Weaver [options] <path>+)\n" +
            "options:\n" +
            "-classpath (-cp) <classpath>\n" +
            "-verbose\n" +
            "-suppress\n" +
            "-outputPath <path>\n" +
            "-outputToJar" +
            "--SOURCEPATH  <source-path-file>\n");
      
      System.out.println(usage);
   }
}
