/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.weaver;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.Deflater;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.NotFoundException;
import javassist.bytecode.BadBytecode;
import javassist.bytecode.ClassFile;

/**
 * Takes jar or class files and weaves the bytecode based on the 
 * given configuration.  A weaver object should be created, and
 * passed to the setWeaver() method.  The weaver object is what
 * does the actual byte code weaving.
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 * @author pgier
 * @version $Revision: 166 $
 */
public class WeaveRunner
{
   private static Logger log = Logger.getLogger(WeaveRunner.class.getName());

   /**
    * The paths in the classpath
    */
   private ArrayList<String> sourcePaths = new ArrayList<String>();

   /**
    * The set of class files to be translated.
    */
   private Set<ClassFileInfo> classesToCompile = new HashSet<ClassFileInfo>();

   private Weaver weaver;

   private boolean verbose = false;

   private boolean suppress = true;

   private String outputPath;

   private boolean outputToJar = false;

   private JarOutputStream jarOutputStream;

   private FileFilter classFileFilter = new FileFilter()
   {
      public boolean accept(File file)
      {
         return file.getName().endsWith(".class");
      }
   };

   private FileFilter directoryFilter = new FileFilter()
   {
      public boolean accept(File file)
      {
         return file.isDirectory();
      }
   };

   /** 
    * Checks if a file is a jar by attempting to open it and read the jar
    * info.
    */
   private FileFilter jarFileFilter = new FileFilter()
   {
      public boolean accept(File file)
      {
         boolean result = false;
         try
         {
            new JarFile(file);
            result = true;
         }
         catch (IOException ioe)
         {
            // Means this is not a jar file.
         }
         return result;
      }
   };

   // Make public and static so that transformers can locate it to do work
   // transformers may generate class files and they need to determine
   // file locations and such.  This will also be used as a flag to tell
   // transformers whether they are in compile or load-time mode.
   public static URLClassLoader loader;

   /**
    * Creates a weave runner with a default Weaver (org.jboss.weaver.Weaver).
    */
   public WeaveRunner()
   {
      this.weaver = new Weaver();
   }

   public WeaveRunner(Weaver weaver)
   {
      this.weaver = weaver;
   }

   public void weave() throws NotFoundException, IOException
   {

      ClassPool pool = ClassPool.getDefault();
      try 
      {
         pool.appendPathList(weaver.getClasspath());
      }
      catch (NotFoundException nfe) 
      {
         throw new IOException(nfe.getMessage());
      }

      URL[] urls = weaver.getClassPathUrls().toArray(new URL[weaver.getClassPathUrls().size()]);
      loader = new URLClassLoader(urls, Thread.currentThread().getContextClassLoader());
      Thread.currentThread().setContextClassLoader(loader);

      // Generate the list of classes to compile
      for (String fileName : sourcePaths)
      {
         File file = new File(fileName);
         if (file.isDirectory())
         {
            addDirectory(file, pool);
         }
         else if (classFileFilter.accept(file))
         {
            addFile(file, pool);
         }
         else if (jarFileFilter.accept(file))
         {
            addJarFile(new JarFile(file), pool);
         }
         else
         {
            log.fine("[retro] " + file + " is neither a java class or a directory");
         }
      }

      if (outputToJar)
      {

         Manifest manifest = new Manifest();
         Attributes attributes = manifest.getMainAttributes();
         attributes.putValue("Manifest-Version", "1.0");
         attributes.putValue("Created-By", System.getProperty("java.vm.version") + " ("
               + System.getProperty("java.vm.vendor") + ")");
         jarOutputStream = new JarOutputStream(new FileOutputStream(outputPath), manifest);
         jarOutputStream.setLevel(Deflater.BEST_COMPRESSION);

      }
      //Compile each class
      for (ClassFileInfo classFileInfo : classesToCompile)
      {
         try 
         {
            compileFile(classFileInfo);
            writeClassFile(classFileInfo);
         }
         catch (CannotCompileException e)
         {
            log.warning("Unable to compile file: " + classFileInfo.getFileName());
            log.warning(e.getMessage());
         }
         catch (BadBytecode e) {
            log.warning("Unable to compile file: " + classFileInfo.getFileName());
            log.warning(e.getMessage());
         }
      }
      if (outputToJar)
      {
         jarOutputStream.close();
      }
   }

   private void writeClassFile(ClassFileInfo classFileInfo) throws IOException
   {
      try
      {
         byte[] classData = classFileInfo.getClazz().toBytecode();
         String classname = classFileInfo.getClassName();
         String filename = this.outputPath + File.separatorChar + classname.replace('.', File.separatorChar) + ".class";
         int pos = filename.lastIndexOf(File.separatorChar);
         if (pos > 0)
         {
            String dir = filename.substring(0, pos);
            if (!dir.equals("."))
            {
               new File(dir).mkdirs();
            }
         }
         if (outputToJar)
         {
            JarEntry jarEntry = new JarEntry(classname.replace('.', File.separatorChar) + ".class");
            jarOutputStream.putNextEntry(jarEntry);
            jarOutputStream.write(classData);
         }
         else
         {
            FileOutputStream fos = new FileOutputStream(filename);
            fos.write(classData);
            fos.close();
         }
      }
      catch (CannotCompileException cce)
      {
         cce.printStackTrace();
      }
   }

   private void addDirectory(File dir, ClassPool pool) throws IOException, NotFoundException
   {
      File[] classFiles = dir.listFiles(classFileFilter);
      for (File classFile : classFiles)
      {
         addFile(classFile, pool);
      }
      File[] directories = dir.listFiles(directoryFilter);
      for (File directory : directories)
      {
         addDirectory(directory, pool);
      }
   }

   private void addJarFile(JarFile jarFile, ClassPool pool) throws IOException, NotFoundException
   {
      Enumeration<JarEntry> jarEntries = jarFile.entries();
      while (jarEntries.hasMoreElements())
      {
         JarEntry nextEntry = jarEntries.nextElement();
         addToCompileList(jarFile.getInputStream(nextEntry), pool);
      }
   }

   private void addFile(File file, ClassPool pool) throws IOException, NotFoundException
   {
      FileInputStream fis = new FileInputStream(file);
      String className = getClassName(fis);
      addToCompileList(className, pool);
   }

   private void addToCompileList(InputStream is, ClassPool pool) throws IOException, NotFoundException
   {
      String className = getClassName(is);
      addToCompileList(className, pool);
   }

   private void addToCompileList(String className, ClassPool pool) throws NotFoundException
   {
      CtClass clazz = pool.get(className);
      classesToCompile.add(new ClassFileInfo(className, clazz));
   }

   /**
    * Get the name of a class from a given input stream
    * @param is The stream with the class file data
    * @return The name of the class
    * @throws IOException
    */
   public String getClassName(InputStream is) throws IOException
   {
      return createClassFile(is).getName();
   }

   public ClassFile createClassFile(InputStream is) throws IOException
   {
      DataInputStream dis = null;
      if (is instanceof DataInputStream)
      {
         dis = (DataInputStream) is;
      }
      else
      {
         dis = new DataInputStream(is);
      }
      ClassFile cf = new ClassFile(dis);
      dis.close();
      return cf;
   }

   public void addFilesFromSourcePathFile(String sourcePathFile)
   {
      BufferedReader reader = null;

      try
      {
         reader = new BufferedReader(new FileReader(new File(sourcePathFile).getCanonicalFile()));

         String fileName = reader.readLine();
         while (fileName != null)
         {
            sourcePaths.add(fileName);
            fileName = reader.readLine();
         }
      }
      catch (IOException ioe)
      {
         log.warning("Problem reading sourcepath file: " + ioe);
         try
         {
            reader.close();
         }
         catch (IOException ioe1)
         {
            throw new RuntimeException(ioe);
         }
      }
   }

   public void compileFile(ClassFileInfo info) throws BadBytecode, CannotCompileException
   {
      if (info.isCompiled())
         return;

      log.fine("[compileFile] " + info.getClassName());
      
      URL classUrl = loader.getResource(info.getClassName().replace('.', '/') + ".class");
      if (classUrl == null)
      {
         System.out.println("[warning] Unable to find " + info.getClassName()
               + " within classpath.  Make sure all transforming classes are within classpath.");
         return;
      }

      // TODO: Is this part necessary?
      /*File classUrlFile = new File(URLDecoder.decode(classUrl.getFile(), "UTF-8"));
       File infoFile = new File(URLDecoder.decode(info.getFile().toString(), "UTF-8"));

       if (!classUrlFile.equals(infoFile))
       {
       System.out.println("[warning] Trying to compile " + info.getFile() + " and found it also within "
       + classUrl.getFile() + "; proceeding. ");
       }*/

      boolean weaved = weaver.doWeave(loader, info);

      if (weaved == false)
      {
         log.fine("[no comp needed] " + info.getFileName());
         return;
      }
      info.setCompiled(true);
      log.fine("[compiled] " + info.getFileName());
   }

   public String getOutputPath()
   {
      return outputPath;
   }

   public void setOutputPath(String outputPath)
   {
      this.outputPath = outputPath;
   }

   public void addSourcePath(String path) throws IOException
   {
      sourcePaths.add(path);
   }

   public ArrayList<String> getSourcePaths()
   {
      return sourcePaths;
   }

   public void setSourcePaths(ArrayList<String> sourcePaths)
   {
      this.sourcePaths = sourcePaths;
   }

   public Weaver getWeaver()
   {
      return weaver;
   }

   public void setWeaver(Weaver weaver)
   {
      this.weaver = weaver;
   }

   public void setWeaver(String weaverClass)
   {
      try
      {
         weaver = (Weaver) Class.forName(weaverClass).newInstance();
      }
      catch (Exception e)
      {
         log.warning("Unable to instantiate weaver: " + weaverClass);
         log.warning(e.getMessage());
      }

   }

   public boolean isOutputToJar()
   {
      return outputToJar;
   }

   public void setOutputToJar(boolean outputToJar)
   {
      this.outputToJar = outputToJar;
   }

   public boolean isSuppress()
   {
      return suppress;
   }

   public void setSuppress(boolean suppress)
   {
      this.suppress = suppress;
   }

   public boolean isVerbose()
   {
      return verbose;
   }

   public void setVerbose(boolean verbose)
   {
      this.verbose = verbose;
      if (verbose) 
      {
         Logger weaverLog = Logger.getLogger("org.jboss.weaver");
         weaverLog.setUseParentHandlers(false);
         Handler console = new ConsoleHandler();
         console.setLevel(Level.FINE);
         weaverLog.addHandler(console);
         weaverLog.setLevel(Level.FINE); 
      }
   }

}
