/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.weaver;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import javassist.CannotCompileException;
import javassist.CodeConverter;
import javassist.CtClass;
import javassist.bytecode.BadBytecode;
import javassist.bytecode.ClassFile;
import javassist.expr.ExprEditor;

/**
 * This class contains the required configuration for weaving
 * @author pgier
 *
 */
public class Weaver
{

   static Logger log = Logger.getLogger(Weaver.class.getName());
   
   /**
    * Path to properties file that contains the mapping between old and new classes.
    */
   private Map<String, String> classRenames = new Hashtable<String, String>();;
   
   /**
    * List of code converters to run against the classes.
    */
   private List<CodeConverter> codeConverters = new ArrayList<CodeConverter>();
   
   /**
    * List of expression editors to run against the classes.
    */
   private List<ExprEditor> exprEditors = new ArrayList<ExprEditor>();

   
   private String classpath = "";

   private ArrayList<URL> classPathUrls = new ArrayList<URL>();
   
   public boolean doWeave(ClassLoader cl, ClassFileInfo info) throws BadBytecode, CannotCompileException
   {
      CtClass clazz = info.getClazz();
      ClassFile file = clazz.getClassFile();

      // Rename known classes
      file.renameClass(classRenames);

      // Run the converters
      for (CodeConverter converter : this.codeConverters)
      {
         clazz.instrument(converter);
      }

      // Run the editors
      for (ExprEditor editor : this.exprEditors)
      {
         clazz.instrument(editor);
      }
      
      return true;
   }
   
   /**
    * Get the class renames.
    * @return The mapping between the original class names and the new class name
    */
   public Map<String, String> getClassRenames()
   {
      return classRenames;
   }

   public void setClassRenames(Map<String, String> classRenames)
   {
      this.classRenames = classRenames;
   }
   
   /**
    * Load the class renames from a properties file.
    * @param props - The InputStream from which to load the properties.
    */
   public void loadClassRenames(InputStream props) {
      Properties classRenamesProps = new Properties();
      try {
         classRenamesProps.load(props);
         this.putClassRenames((Map)classRenamesProps);
      } catch (IOException ioe) {
         log.warning("Unable to load class renames from given input stream");
         log.warning(ioe.getMessage());
         if (log.getLevel() == Level.FINE) {
            ioe.printStackTrace();
         }
      }
   }
   
   public void putClassRename(String origClass, String renamedClass) {
      classRenames.put(origClass, renamedClass);
   }
   
   public void putClassRenames(Map<String, String> props) {
      this.classRenames.putAll(props);
   }
   
   /** 
    * Clear the class renames.
    */
   public void clearClassRenames() {
      classRenames.clear();
   }
      
   public void addCodeConverter(CodeConverter converter) {
      codeConverters.add(converter);
   }
   
   public List<CodeConverter> getCodeConverters() {
      return codeConverters;
   }
   
   public void clearCodeConverters() {
      codeConverters.clear();
   }
   
   public void addExprEditor(ExprEditor editor) {
      exprEditors.add(editor);
   }
   
   public List<ExprEditor> getExprEditors() {
      return exprEditors;
   }
   
   public void clearExprEditors() {
      exprEditors.clear();
   }

   public String getClasspath()
   {
      return classpath;
   }

   /**
    * Set the classpath.
    * @param classPath to be used by the weaver
    */
   public void setClasspath(String classPath)
   {
      if (classPath == null) {
         return;
      }
      StringTokenizer tokenizer = new StringTokenizer(classPath, File.pathSeparator);
      while (tokenizer.hasMoreTokens())
      {
         String cpath = tokenizer.nextToken();
         File f = new File(cpath);
         try
         {
            classPathUrls.add(f.toURL());
         }
         catch (MalformedURLException mue)
         {
            throw new IllegalArgumentException("Malformed URL in classpath param: " + mue);
         }
      }

      this.classpath = classPath;
   }

   public ArrayList<URL> getClassPathUrls()
   {
      return classPathUrls;
   }

   public void setClassPathUrls(ArrayList<URL> classPathUrls)
   {
      this.classPathUrls = classPathUrls;
   }
   
   /**
    * Post construction initialization.
    * Subclasses should override this to perform their own init.
    */
   public void init() {
      
   }

}
