/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.weaver.retro;

import java.io.StringWriter;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.HashSet;
import java.util.Timer;
import java.util.regex.Matcher;

import javassist.CannotCompileException;
import javassist.expr.ExprEditor;
import javassist.expr.FieldAccess;
import javassist.expr.MethodCall;
import javassist.expr.NewExpr;

import org.jboss.lang.AnnotationHelper;
import org.jboss.lang.CharacterRedirects;
import org.jboss.lang.ClassRedirects;
import org.jboss.lang.ExceptionHelper;
import org.jboss.lang.GenericsHelper;
import org.jboss.lang.JBossStringBuilder;
import org.jboss.lang.NumberHelper;
import org.jboss.lang.StringRedirects;
import org.jboss.util.StringWriterHelper;

import edu.emory.mathcs.backport.java.util.Collections;
import edu.emory.mathcs.backport.java.util.concurrent.TimeUnit;
import edu.emory.mathcs.backport.java.util.concurrent.helpers.Utils;
import edu.emory.mathcs.backport.java.util.concurrent.locks.Condition;

/**
 * An expression editor implementation that maps news, field access and method
 * calls that exist in java5 to java14 compatible helper implementations.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author Scott.Stark@jboss.org
 * @version $Revision: 200 $
 */
public class ClassRedirectEditor extends ExprEditor
{
   /** Class redirects */
   private static final String CLASS_REDIRECTS = ClassRedirects.class.getName();
   /** String redirects */
   private static final String STRING_REDIRECTS = StringRedirects.class.getName();
   /** Character redirects */
   private static final String CHARACTER_REDIRECTS = CharacterRedirects.class.getName();
   /** Annotation helper */
   private static final String ANNOTATION_HELPER = AnnotationHelper.class.getName();
   /** Exception helper */
   private static final String EXCEPTION_HELPER = ExceptionHelper.class.getName();
   /** StringWriter helper */
   private static final String STRINGWRITER_HELPER = StringWriterHelper.class.getName();
   /** Concurrent utils */
   private static final String CONCURRENT_UTILS = Utils.class.getName();
   /** Integer redirects */
   private static final String NUMBER_HELPER = NumberHelper.class.getName();
   /** Generics Helper */
   private static final String GENERICS_HELPER = GenericsHelper.class.getName();

   /** Object class */
   private static final String OBJECT = Object.class.getName();
   /** Class class */
   private static final String CLASS = Class.class.getName();
   /** Method class */
   private static final String METHOD = Method.class.getName();
   /** Constructor class */
   private static final String CONSTRUCTOR = Constructor.class.getName();
   /** Field class */
   private static final String FIELD = Field.class.getName();
   /** AccessibleObject class */
   private static final String ACCESSIBLE_OBJECT = AccessibleObject.class.getName();
   /** String class */
   private static final String STRING = String.class.getName();
   /** Boolean class */
   private static final String BOOLEAN = Boolean.class.getName();
   /** Integer class */
   private static final String INTEGER = Integer.class.getName();
   /** Long class */
   private static final String LONG = Long.class.getName();
   /** Character class */
   private static final String CHARACTER = Character.class.getName();
   /** Matcher class */
   private static final String MATCHER = Matcher.class.getName();
   /** System class */
   private static final String SYSTEM = System.class.getName();
   /** Collections class */
   private static final String COLLECTIONS = Collections.class.getName();
   /** ThreadLocal class */
   private static final String THREAD_LOCAL = ThreadLocal.class.getName();
   /** InheritableThreadLocal class */
   private static final String INHERTIABLE_THREAD_LOCAL = InheritableThreadLocal.class.getName();
   /** Condition class */
   private static final String CONDITION = Condition.class.getName();
   /** TimeUnit class */
   private static final String TIME_UNIT = TimeUnit.class.getName();
   /** StringWriter class */
   private static final String STRING_WRITER = StringWriter.class.getName();
   /** Timer class */
   private static final String TIMER = Timer.class.getName();
   /** URL class */
   private static final String URL = URL.class.getName();
   /** StringBuffer class */
   private static final String STRING_BUFFER = StringBuffer.class.getName();
   /** StringBuilder class */
   private static final String STRING_BUILDER = JBossStringBuilder.class.getName();

   /** The new jdk5 exception ctor signatures */
   private static HashSet<String> exceptionCtors = new HashSet<String>();
   static {
      exceptionCtors.add("java.security.cert.CertificateException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.cert.CertificateException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.InvalidKeyException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.InvalidKeyException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.KeyManagementException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.KeyManagementException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.SignatureException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.SignatureException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("org.xml.sax.SAXException()V");
      exceptionCtors.add("org.xml.sax.SAXNotRecognizedException()V");
      exceptionCtors.add("java.security.cert.CRLException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.cert.CRLException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.cert.CertificateEncodingException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.cert.CertificateEncodingException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.lang.IllegalArgumentException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.lang.IllegalArgumentException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.GeneralSecurityException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.GeneralSecurityException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.cert.CertificateParsingException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.cert.CertificateParsingException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.KeyStoreException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.KeyStoreException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.ProviderException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.ProviderException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.InvalidAlgorithmParameterException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.InvalidAlgorithmParameterException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.lang.IllegalStateException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.lang.IllegalStateException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.lang.UnsupportedOperationException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.lang.UnsupportedOperationException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.lang.SecurityException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.lang.SecurityException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.DigestException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.DigestException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("org.xml.sax.SAXNotSupportedException()V");
      exceptionCtors.add("java.security.NoSuchAlgorithmException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.NoSuchAlgorithmException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.spec.InvalidKeySpecException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.spec.InvalidKeySpecException(Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.KeyException(Ljava/lang/String;Ljava/lang/Throwable;)V");
      exceptionCtors.add("java.security.KeyException(Ljava/lang/Throwable;)V");
   }
   private static HashSet<String> integerBitMethods = new HashSet<String>();
   static {
     String s[] = new String[] { "highestOneBit", "lowestOneBit", "numberOfLeadingZeros",
               "bitCount", "rotateLeft", "rotateRight", "reverse", "signum",
               "reverseBytes" };
     for (int i = 0; i < s.length; i++)
       integerBitMethods.add(s[i]); 
   }
   private static HashSet<String> characterCodePointMethods = new HashSet<String>();
   static {
      characterCodePointMethods.add("charCount");
      characterCodePointMethods.add("codePointAt");
      characterCodePointMethods.add("codePointBefore");
      characterCodePointMethods.add("codePointCount");
      characterCodePointMethods.add("isHighSurrogate");
      characterCodePointMethods.add("isLowSurrogate");
      characterCodePointMethods.add("isSurrogatePair");
      characterCodePointMethods.add("offsetByCodePoints");
      characterCodePointMethods.add("toChars");
      characterCodePointMethods.add("toCodePoint");
      characterCodePointMethods.add("reverseBytes");
   }
   private static HashSet<String> characterOverloadMethods = new HashSet<String>();
   static {
      characterOverloadMethods.add("isLowerCase(I)Z");
      characterOverloadMethods.add("isUpperCase(I)Z");
      characterOverloadMethods.add("isTitleCase(I)Z");
      characterOverloadMethods.add("isDigit(I)Z");
      characterOverloadMethods.add("isDefined(I)Z");
      characterOverloadMethods.add("isLetter(I)Z");
      characterOverloadMethods.add("isLetterOrDigit(I)Z");
      characterOverloadMethods.add("isJavaIdentifierStart(I)Z");
      characterOverloadMethods.add("isJavaIdentifierPart(I)Z");
      characterOverloadMethods.add("isUnicodeIdentifierStart(I)Z");
      characterOverloadMethods.add("isUnicodeIdentifierPart(I)Z");
      characterOverloadMethods.add("isUnicodeIdentifierIgnorable(I)Z");
      characterOverloadMethods.add("toLowerCase(I)I");
      characterOverloadMethods.add("toUpperCase(I)I");
      characterOverloadMethods.add("toTitleCase(I)I");
      characterOverloadMethods.add("digit(I)I");
      characterOverloadMethods.add("getNumericValue(I)I");
      characterOverloadMethods.add("isSpaceChar(I)Z");
      characterOverloadMethods.add("isWhitespace(I)Z");
      characterOverloadMethods.add("isISOControl(I)Z");
      characterOverloadMethods.add("getType(I)I");
      characterOverloadMethods.add("getDirectionality(I)B");
      characterOverloadMethods.add("isMirrored(I)Z");
      characterOverloadMethods.add("reverseBytes(C)C");
   }
   private static HashSet<String> bufferBuilderCodePointMethods = new HashSet<String>();
   static {
      bufferBuilderCodePointMethods.add("appendCodePoint");
      bufferBuilderCodePointMethods.add("codePointAt");
      bufferBuilderCodePointMethods.add("codePointBefore");
      bufferBuilderCodePointMethods.add("codePointCount");
      bufferBuilderCodePointMethods.add("offsetByCodePoints");
   }
   private static final String CTOR_TIMER1 = TIMER + "(Ljava/lang/String;)V";
   private static final String CTOR_TIMER2 = TIMER + "(Ljava/lang/String;Z)V";

   /**
    *
    *  @param newExpr
    *  @throws CannotCompileException
    */
   public void edit(NewExpr newExpr) throws CannotCompileException
   {
      String signature = newExpr.getClassName() + newExpr.getSignature();
      if (exceptionCtors.contains(signature))
      {
         newExpr.replace("$_ = " + EXCEPTION_HELPER + ".createExceptionWithCause($class, $args);");
      }
      else if (CTOR_TIMER1.equals(signature))
      {
         newExpr.replace("$_ = new " + TIMER + "();");
      }
      else if (CTOR_TIMER2.equals(signature))
      {
         newExpr.replace("$_ = new " + TIMER + "($2);");
      }
      
   }

   public void edit(FieldAccess access) throws CannotCompileException
   {
      if (COLLECTIONS.equals(access.getClassName()))
      {
         if ("EMPTY_LIST".equals(access.getFieldName()))
            access.replace("$_ = " + COLLECTIONS + ".emptyList();");
         else if ("EMPTY_SET".equals(access.getFieldName()))
            access.replace("$_ = " + COLLECTIONS + ".emptySet();");
         else if ("EMPTY_MAP".equals(access.getFieldName()))
            access.replace("$_ = " + COLLECTIONS + ".emptyMap();");
      }
   }

   public void edit(MethodCall call) throws CannotCompileException
   {
      String callName = call.getMethodName();
      String className = call.getClassName();
      if (CLASS.equals(className))
      {
         // clazz.isEnum() -> ClassRedirects.isEnum(clazz)
         if ("isEnum".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".isEnum($0);");
         // clazz.getEnumConstants - > ClassRedirects.values(clazz)
         if ("isAnnotation".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".isAnnotation($0);");
         // clazz.getEnumConstants - > ClassRedirects.values(clazz)
         else if ("getEnumConstants".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".getEnumConstants($0);");
         // clazz.cast(o) -> ClassRedirects.cast(clazz, o)
         else if ("cast".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".cast($0, $1);");
         // clazz.asSubclass(o) -> ClassRedirects.cast(clazz, o)
         else if ("asSubclass".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".asSubclass($0, $1);");
         // clazz.getSimpleName() -> ClassRedirects.getSimpleName(clazz)
         else if ("getSimpleName".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".getSimpleName($0);");
         // clazz.getCanonicalName() -> ClassRedirects.getCanonicalName(clazz)
         else if ("getCanonicalName".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".getCanonicalName($0);");
         // clazz.isAnonymousClass() -> ClassRedirects.isAnonymousClass(clazz)
         else if ("isAnonymousClass".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".isAnonymousClass($0);");
         // clazz.getGenericSuperclass() -> ClassRedirects.getGenericSuperclass(clazz)
         else if ("getGenericSuperclass".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".getGenericSuperclass($0);");
         // clazz.getGenericInterfaces() -> ClassRedirects.getGenericInterfaces(clazz)
         else if ("getGenericInterfaces".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".getGenericInterfaces($0);");
      }
      else if (METHOD.equals(className))
      {
         // method.getParameterAnnotations() -> AnnotationHelper.getParameterAnnotations(method)
         if ("getParameterAnnotations".equals(callName))
            call.replace("$_ = " + ANNOTATION_HELPER + ".getParameterAnnotations($0);");
         // method.getGenericReturnType() -> GenericHelper.getGenericReturnType(method)
         else if ("getGenericReturnType".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".getGenericReturnType($0);");
         // method.getGenericReturnType() -> GenericHelper.getGenericReturnType(method)
         else if ("getGenericReturnType".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".getGenericReturnType($0);");
         // method.getGenericParameterTypes() -> GenericHelper.getGenericParameterTypes(method)
         else if ("getGenericParameterTypes".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".getGenericParameterTypes($0);");
         // method.getGenericExceptionTypes() -> GenericHelper.getGenericExceptionTypes(method)
         else if ("getGenericExceptionTypes".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".getGenericExceptionTypes($0);");
         // method.toGenericString() -> GenericHelper.toGenericString(method)
         else if ("toGenericString".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".toGenericString($0);");
      }
      else if (FIELD.equals(className))
      {
         // field.getGenericType() -> GenericHelper.getGenericType(field)
         if ("getGenericType".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".getGenericType($0);");
         // field.toGenericString() -> GenericHelper.toGenericString(field)
         else if ("toGenericString".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".toGenericString($0);");
      }
      else if (CONSTRUCTOR.equals(className))
      {
         // constructor.getParameterAnnotations() -> AnnotationHelper.getParameterAnnotations(constructor)
         if ("getParameterAnnotations".equals(callName))
            call.replace("$_ = " + ANNOTATION_HELPER + ".getParameterAnnotations($0);");
         // constructor.getGenericParameterTypes() -> GenericHelper.getGenericParameterTypes(constructor)
         else if ("getGenericParameterTypes".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".getGenericParameterTypes($0);");
         // constructor.getGenericExceptionTypes() -> GenericHelper.getGenericExceptionTypes(constructor)
         else if ("getGenericExceptionTypes".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".getGenericExceptionTypes($0);");
         // constructor.toGenericString() -> GenericHelper.toGenericString(constructor)
         else if ("toGenericString".equals(callName))
            call.replace("$_ = " + GENERICS_HELPER + ".toGenericString($0);");
      }
      else if (STRING.equals(className))
      {
         String signature = call.getSignature();
         if ("replace".equals(callName) && signature.equals("(Ljava/lang/CharSequence;Ljava/lang/CharSequence;)Ljava/lang/String;"))
            call.replace("$_ = " + STRING_REDIRECTS + ".replace($0, $1, $2);");
         else if ("contains".equals(callName))
            call.replace("$_ = " + STRING_REDIRECTS + ".contains($0, $1);");
         else if ("codePointAt".equals(callName))
            call.replace("$_ = " + STRING_REDIRECTS + ".codePointAt($0, $1);");
         else if ("codePointBefore".equals(callName))
            call.replace("$_ = " + STRING_REDIRECTS + ".codePointBefore($0, $1);");
         else if ("codePointCount".equals(callName))
            call.replace("$_ = " + STRING_REDIRECTS + ".codePointCount($0, $1, $2);");
         else if ("offsetByCodePoints".equals(callName))
            call.replace("$_ = " + STRING_REDIRECTS + ".offsetByCodePoints($0, $1, $2);");
      }
      else if (STRING_BUFFER.equals(className) || STRING_BUILDER.equals(className))
      {
         if (bufferBuilderCodePointMethods.contains(callName))
            call.replace("$_ = " + STRING_REDIRECTS + "." + callName + "($0, $$);");
      }
      else if (BOOLEAN.equals(className))
      {
          if (callName.equals("parseBoolean")) 
          {
               call.replace("$_ = " + org.jboss.lang.Boolean.class.getName() + "." + callName + "($$);");
          }
          else if (callName.equals("compareTo"))
          {
               call.replace("$_ = " + org.jboss.lang.Boolean.class.getName() + "." + callName + "($0, $1);");
          }
      }
      else if (INTEGER.equals(className))
      {
       if (integerBitMethods.contains(callName))
            call.replace("$_ = " + NUMBER_HELPER + "." + callName + "($$);");
      }
      else if (LONG.equals(className))
      {
       if (integerBitMethods.contains(callName))
            call.replace("$_ = " + NUMBER_HELPER + "." + callName + "($$);");
      }
      else if (CHARACTER.equals(className))
      {
         if (characterCodePointMethods.contains(callName))
         {
            call.replace("$_ = " + CHARACTER_REDIRECTS + "." + callName + "($$);");
         }
         else
         {
            String signature = callName + call.getSignature();
            if (characterOverloadMethods.contains(signature))
            {
               call.replace("$_ = " + CHARACTER_REDIRECTS + "." + callName + "($$);");
            }
         }
      }
      else if (MATCHER.equals(className))
      {
         if ("quoteReplacement".equals(callName))
            call.replace("$_ = " + CLASS_REDIRECTS + ".quoteReplacement($1);");
      }
      else if (SYSTEM.equals(className))
      {
         if ("nanoTime".equals(callName))
            call.replace("$_ = " + CONCURRENT_UTILS + ".nanoTime();");
      }
      else if (THREAD_LOCAL.equals(className))
      {
         if ("remove".equals(callName))
            call.replace("$0.set(null);");
      }
      else if (INHERTIABLE_THREAD_LOCAL.equals(className))
      {
         if ("remove".equals(callName))
            call.replace("$0.set(null);");
      }
      else if (CONDITION.equals(className))
      {
         if ("awaitNanos".equals(callName))
            call.replace("$_ = $0.await($1, " + TIME_UNIT + ".NANOSECONDS);");
      }
      else if (STRING_WRITER.equals(className))
      {
         if ("append".equals(callName))
         {
            String signature = call.getSignature();
            if( signature.equals("(Ljava/lang/CharSequence;)Ljava/io/StringWriter;") )
               call.replace("$_ = " + STRINGWRITER_HELPER + ".append($0, $1);");
            else if( signature.equals("(C)Ljava/io/StringWriter;") )
               call.replace("$_ = " + STRINGWRITER_HELPER + ".append($0, $1);");
            else if( signature.equals("(Ljava/lang/CharSequence;II)Ljava/io/StringWriter;") )
               call.replace("$_ = " + STRINGWRITER_HELPER + ".append($0, $1, $2, $3);");
         }
      }
      else if (URL.equals(className))
      {
         if ("toURI".equals(callName))
         {
            call.replace("$_ = java.net.URI.create(" + "$0.toString());");
         }
      }
      else if (TIMER.equals(className))
      {
         if ("purge".equals(callName))
         {
            call.replace("$_ = 0;");
         }
      }
      
      if (OBJECT.equals(className) || METHOD.endsWith(className) || CLASS.equals(className) || FIELD.equals(className)
            || CONSTRUCTOR.equals(className)) {
         if ("getTypeParameters".equals(callName)) {
            call.replace("$_ = " + GENERICS_HELPER + ".getTypeParameters($0);");
         }
      }
      
      // Replace calls to annotation methods
      if (OBJECT.equals(className) || METHOD.endsWith(className) || CLASS.equals(className) || FIELD.equals(className)
            || CONSTRUCTOR.equals(className) || ACCESSIBLE_OBJECT.equals(className))
      {
         if ("isAnnotationPresent".equals(callName)) 
         {
            call.replace("$_ = " + ANNOTATION_HELPER + ".isAnnotationPresent($0, $1);");
            // obj.isAnnotationPresent(annotation) -> AnnotationHelper.isAnnotationPresent(obj, annotationType)
         }
         else if ("getAnnotations".equals(callName))
         {
            // obj.getAnnotations() -> AnnotationHelper.getAnnotations(obj)
            call.replace("$_ = " + ANNOTATION_HELPER + ".getAnnotations($0);");
         }
         else if ("getAnnotation".equals(callName))
         {
            // obj.getAnnotation(annotationType) -> AnnotationHelper.getAnnotation(obj, annotationType)
            call.replace("$_ = " + ANNOTATION_HELPER + ".getAnnotation($0, $1);");
         }
         else if ("getDeclaredAnnotations".equals(callName))
         {
            // obj.getDeclaredAnnotations() -> AnnotationHelper.getDeclaredAnnotations(obj)
            call.replace("$_ = " + ANNOTATION_HELPER + ".getDeclaredAnnotations($0);");
         }
      }
   }
}
