/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ha.framework.server;

import java.net.InetAddress;

import org.jboss.cache.Cache;
import org.jboss.cache.config.Configuration;
import org.jboss.ha.framework.interfaces.DistributedState;
import org.jboss.system.server.ServerConfigUtil;
import org.jgroups.ChannelFactory;

/**
 * Configuration POJO for {@link ClusterPartition}.
 *   
 * @author Brian Stansberry
 *   
 * @version $Revision: 56922 $
 */
public class ClusterPartitionConfig
{
   // Constants -----------------------------------------------------

   // Attributes ----------------------------------------------------

   private   Cache cache;
   private   ChannelFactory multiplexer;
   private   DistributedState distributedState;
   private   String stackName;
   private   String partitionName = ServerConfigUtil.getDefaultPartitionName();
   private   boolean deadlock_detection = false;
   private   boolean allow_sync_events = false;
   private   String nodeUniqueId = null;
   private   InetAddress nodeAddress = null;
   private   int namingServicePort = -1;

   /** Number of milliseconds to wait until state has been transferred. Increase this value for large states
    * 0 = wait forever
    */
   private   long state_transfer_timeout=60000;


   private   long method_call_timeout=60000;

   // Static --------------------------------------------------------

   // Constructors --------------------------------------------------

   // Public --------------------------------------------------------

   public String getPartitionName()
   {
      return partitionName;
   }

   public void setPartitionName(String newName)
   {
      partitionName = newName;
   }

   /**
    * Uniquely identifies this node. MUST be unique accros the whole cluster!
    * Cannot be changed once the partition has been started
    */
   public String getNodeUniqueId()
   {
      return this.nodeUniqueId;
   }

   public void setNodeUniqueId(String node)
   {
      this.nodeUniqueId = node;
   }

   public InetAddress getNodeAddress()
   {
      return nodeAddress;
   }

   public void setNodeAddress(InetAddress address)
   {
      this.nodeAddress = address;
   }

   public long getStateTransferTimeout()
   {
      return state_transfer_timeout;
   }

   public void setStateTransferTimeout(long timeout)
   {
      this.state_transfer_timeout=timeout;
   }

   public long getMethodCallTimeout() {
      return method_call_timeout;
   }

   public void setMethodCallTimeout(long timeout) {
      this.method_call_timeout=timeout;
   }
   
   public boolean getDeadlockDetection()
   {
      return deadlock_detection;
   }

   public void setDeadlockDetection(boolean doit)
   {
      deadlock_detection = doit;
   }

   public boolean getAllowSynchronousMembershipNotifications()
   {
      return allow_sync_events;
   }

   public void setAllowSynchronousMembershipNotifications(boolean allowSync)
   {
      this.allow_sync_events = allowSync;
   }

   public ChannelFactory getMultiplexer()
   {
      return multiplexer;
   }

   public String getMultiplexerStack()
   {
      return stackName;
   }
   
   public Cache getClusteredCache()
   {
      return cache;
   }
   
   /**
    * Sets the Cache used by this partition for state management.
    * 
    * <strong>NOTE:</strong> The cache must be configured to use a JGroups
    * multiplexer channel.
    * @param cache the cache
    * 
    * @throws IllegalArgumentException if the cache is not configured to use a multiplexer
    * @throws NullPointerException if cache is <code>null</code>
    */
   public void setClusteredCache(Cache cache)
   {
      this.cache = cache;
      Configuration config = cache.getConfiguration();
      multiplexer = config.getRuntimeConfig().getMuxChannelFactory();
      
      if (multiplexer == null)
         throw new IllegalArgumentException("Cache not configured for a multiplexer");
      
      this.stackName = config.getMultiplexerStack();
   }

   public int getNamingServicePort()
   {      
      return namingServicePort;
   }

   public void setNamingServicePort(int namingServicePort)
   {
      this.namingServicePort = namingServicePort;
   }

   public DistributedState getDistributedState()
   {
      return distributedState;
   }

   public void setDistributedState(DistributedState distributedState)
   {
      this.distributedState = distributedState;
   }
}
