/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.singleton;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.List;

import org.jboss.ha.framework.interfaces.ClusterNode;
import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.ServerConfigUtil;

/**
 * A base class for policy service that decides which node in the cluster should be 
 * the master node to run certain HASingleton service.
 * 
 * @author <a href="mailto:afu@novell.com">Alex Fu</a>
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a> 
 * @version $Revision: 69598 $
 */
public abstract class HASingletonElectionPolicyBase 
   extends ServiceMBeanSupport
   implements HASingletonElectionPolicyMBean
{
   private String mSingletonName;
   private HAPartition mPartition;
   private String mPreferredMaster;
   
   /**
    * @see HASingletonElectionPolicy#setSingletonName
    */
   public void setSingletonName(String singletonName)
   {
      this.mSingletonName = singletonName;
      log.debug("set singleton name to " + this.mSingletonName);
   }   
   
   /**
    * @see HASingletonElectionPolicy#getSingletonName
    */
   public String getSingletonName()
   {
      return this.mSingletonName;
   }
   
   /**
    * @see HASingletonElectionPolicy#setPreferredMaster(ClusterNode)
    */
   public void setPreferredMaster(String node)
   {
      this.mPreferredMaster = node;
   }
   
   /**
    * @see HASingletonElectionPolicy#getPreferredMaster()
    */
   public String getPreferredMaster()
   {
      return this.mPreferredMaster;
   }
   
   /**
    * @see HASingletonElectionPolicy#setHAPartition(HAPartition)
    */
   public void setHAPartition(HAPartition partition)
   {
      this.mPartition = partition;
   }

   /**
    * @see HASingletonElectionPolicy#getHAPartition()
    */
   public HAPartition getHAPartition()
   {
      return this.mPartition;
   }
      
   /**
    * @see HASingletonElectionPolicy#elect()
    */
   public ClusterNode elect() 
   {
      List<ClusterNode> candidates = getCandidates();
      if (candidates == null)
      {
         throw new IllegalStateException("list of cluster node candidates where to run the singleton cannot be null");
      }
      
      // If preferred master is defined and contained in cluster, return it
      if (null != this.mPreferredMaster) {
       
         log.debug("Checking if " + mPreferredMaster + " is in candidate list " + candidates);
         
         // First just match on names
         for (ClusterNode node : candidates)
         {
            if (node.getName().equals(this.mPreferredMaster))
            {
               return node;
            }
         }
         
         // No match. Check for situation where preferred master uses a hostname
         // and the ClusterNode uses dotted decimal (or vice versa)
         
         // See if we can parse out an InetAddress and port from mPreferredMaster.
         InetAddress addr = null;
         int port = -1;
         // Look for ':' in the middle of the string
         int idx = mPreferredMaster.indexOf(':');
         if (idx > 0 && idx < mPreferredMaster.length() -1)
         {
            String portString = mPreferredMaster.substring(idx +1);
            String hostName = mPreferredMaster.substring(0, idx);
            try
            {
               port = Integer.parseInt(portString);
               addr = InetAddress.getByName(hostName);
               log.debug("Parsed " + mPreferredMaster + " into " + addr + " and " + port);
            }
            catch (NumberFormatException nfe)
            {
               log.debug(mPreferredMaster + " does not end in an int; cannot parse out a port", nfe);
            }
            catch (UnknownHostException uhe)
            {
               log.debug("Cannot extract InetAddress from " + mPreferredMaster, uhe);
            }
         } 
         
         if (addr != null)
         {
            String rewritten = addr.getHostAddress() + ":" + port;
            if (mPreferredMaster.equals(rewritten))
            {
               rewritten = addr.getHostName() + ":" + port;
            }
            
            for (ClusterNode node : candidates)
            {
               if (node.getName().equals(rewritten))
               {
                  return node;
               }
            }
         }
      }
       
      // Preferred master is not available, election policy is in effect
      return elect(candidates);    
   }
   
   /**
    * Get the list of candidate {@link ClusterNode} instances where the 
    * singleton could be deployed.
    * 
    * @return List of {@link ClusterNode} instances.
    */
   protected List<ClusterNode> getCandidates() 
   {
      return getHAPartition().getDistributedReplicantManager().lookupReplicantsNodes(getSingletonName());
   }
   
   /**
    * Given a List of candidate {@link ClusterNode} instances, return the 
    * elected node where the singleton should run.
    * 
    * @param candidates List of {@link ClusterNode}.
    * @return {@link ClusterNode} instance.
    */
   protected abstract ClusterNode elect(List<ClusterNode> candidates);
   
   private AddressPort parseAddressPort(String nodeName)
   {
      InetAddress addr = null;
      int port = -1;
      // Look for ':' in the middle of the string
      int idx = nodeName.indexOf(':');
      if (idx > 0 && idx < nodeName.length() -1)
      {
         String portString = nodeName.substring(idx +1);
         String hostName = nodeName.substring(0, idx);
         try
         {
            port = Integer.parseInt(portString);
            addr = InetAddress.getByName(hostName);
            return new AddressPort(addr, port);
         }
         catch (NumberFormatException nfe)
         {
            log.debug(nodeName + " does not end in an int; cannot parse out a port", nfe);
         }
         catch (UnknownHostException uhe)
         {
            log.debug("Cannot extract InetAddress from " + nodeName, uhe);
         }
      } 
      return null;
   }
   
   private class AddressPort
   {
      private InetAddress address;
      private int port;
      
      private AddressPort(InetAddress address, int port)
      {
         this.address = address;
         this.port = port;
      }
   }
}
