/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.resource.deployers.management;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.List;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.lang.reflect.UndeclaredThrowableException;

import org.jboss.annotation.factory.AnnotationProxy;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.annotation.ManagementObjectID;
import org.jboss.managed.plugins.DefaultFieldsImpl;
import org.jboss.managed.plugins.ManagedObjectImpl;
import org.jboss.managed.plugins.ManagedPropertyImpl;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.types.SimpleMetaType;
import org.jboss.metatype.api.types.MetaTypeFactory;
import org.jboss.profileservice.management.plugins.BasicDeploymentTemplateInfo;
import org.jboss.resource.metadata.mcf.LocalDataSourceDeploymentMetaData;
import org.jboss.resource.metadata.mcf.XADataSourceDeploymentMetaData;
import org.jboss.resource.metadata.mcf.DBMSMetaData;
import org.jboss.resource.metadata.mcf.NoTxDataSourceDeploymentMetaData;
import org.jboss.resource.metadata.mcf.NoTxConnectionFactoryDeploymentMetaData;
import org.jboss.resource.metadata.mcf.TxConnectionFactoryDeploymentMetaData;

/**
 * The template for creating 
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 69719 $
 */
public class DsDataSourceTemplateInfo extends BasicDeploymentTemplateInfo
{
   private static final long serialVersionUID = 1;
   private static final MetaTypeFactory METATYPE_FACTORY = MetaTypeFactory.getInstance();
   private Map<String, String> propertyNameMappings;
   private String dsType = "local-tx-datasource";
   private transient Type mapType;
   private transient Type mapOfMapsType;

   public DsDataSourceTemplateInfo(String name, String description, String datasourceType)
   {
      super(name, description);
      this.dsType = datasourceType;
   }

   public Map<String, String> getPropertyNameMappings()
   {
      return propertyNameMappings;
   }
   public void setPropertyNameMappings(Map<String, String> propertyNameMappings)
   {
      this.propertyNameMappings = propertyNameMappings;
   }


   public String getConnectionFactoryType()
   {
      return dsType;
   }

   public void setConnectionFactoryType(String dsType)
   {
      this.dsType = dsType;
   }

   public void start()
   {
/*
      // The type of the datasource
      ManagedObjectImpl mo2 = new ManagedObjectImpl(ManagedConnectionFactoryDeploymentGroup.class.getName());
      DefaultFieldsImpl dsTypeField = new DefaultFieldsImpl();
      setFieldName("datasource-type", dsTypeField);
      dsTypeField.setDescription("The type of the DataSource");
      dsTypeField.setMandatory(true);
      dsTypeField.setValue("local-tx-datasource");
      HashSet<MetaValue> values = new HashSet<MetaValue>();
      values.add(SimpleValueSupport.wrap("local-tx-datasource"));
      values.add(SimpleValueSupport.wrap("no-tx-datasource"));
      values.add(SimpleValueSupport.wrap("xa-datasource"));
      values.add(SimpleValueSupport.wrap("no-tx-connection-factory"));
      values.add(SimpleValueSupport.wrap("tx-connection-factory"));
      dsTypeField.setLegalValues(values);
      dsTypeField.setMetaType(SimpleMetaType.STRING);
      ManagedPropertyImpl dsTypeProp = new ManagedPropertyImpl(mo2, dsTypeField);
      super.addProperty(dsTypeProp);

      if(this.dsType != null)
         dsTypeField.setValue(this.dsType);
      else
         this.dsType = (String)dsTypeField.getValue();
*/

      // DataSource
      if("local-tx-datasource".equals(dsType))
         createLocalTxDsTemplate();
      else if("xa-datasource".equals(dsType))
         createXaDsTemplate();
      else if("tx-connection-factory".equals(dsType))
         createTxCfTemplate();
      else if("no-tx-datasource".equals(dsType))
         createNoTxDsTemplate();
      else if("no-tx-connection-factory".equals(dsType))
         createNoTxCfTemplate();
      else
         throw new IllegalStateException("Unsupported dsType: " + dsType);
   }

   private void createXaDsTemplate()
   {
      ManagedObjectImpl mo = new ManagedObjectImpl(XADataSourceDeploymentMetaData.class.getName());
      addDsProperties(mo);
      addManagedProperty("xa-datasource-class", "The xa datasource class name", true, SimpleMetaType.STRING, mo);
      addManagedProperty("xa-datasource-properties", "The xa datasource properties", false, METATYPE_FACTORY.resolve(getMapType()), mo);
      addManagedProperty("url-property", "The URL property", true, SimpleMetaType.STRING, mo);
      addManagedProperty("xa-resource-timeout", "The XA resource timeout", true, SimpleMetaType.INTEGER, new Integer(0), mo);
   }

   private void createLocalTxDsTemplate()
   {
      ManagedObjectImpl mo = new ManagedObjectImpl(LocalDataSourceDeploymentMetaData.class.getName());
      addNonXADsProperties(mo);
   }

   private void createNoTxDsTemplate()
   {
      ManagedObjectImpl mo = new ManagedObjectImpl(NoTxDataSourceDeploymentMetaData.class.getName());
      addNonXADsProperties(mo);
   }

   private void createTxCfTemplate()
   {
      ManagedObjectImpl mo = new ManagedObjectImpl(TxConnectionFactoryDeploymentMetaData.class.getName());
      addCommonProperties(mo);
      addManagedProperty("xa-resource-timeout", "The XA resource timeout", true, SimpleMetaType.INTEGER, new Integer(0), mo);
      addManagedProperty("xa-transaction", "", true, SimpleMetaType.BOOLEAN, mo);
   }

   private void createNoTxCfTemplate()
   {
      ManagedObjectImpl mo = new ManagedObjectImpl(NoTxConnectionFactoryDeploymentMetaData.class.getName());
      addCommonProperties(mo);
   }

   private void addNonXADsProperties(ManagedObjectImpl mo)
   {
      addDsProperties(mo);
      addManagedProperty("driver-class", "The jdbc driver class name", true, SimpleMetaType.STRING, mo);
      addManagedProperty("connection-url", "The jdbc url of the DataSource", true, SimpleMetaType.STRING, mo);
      addManagedProperty("connection-properties", "The jdbc driver connection properties", false, METATYPE_FACTORY.resolve(List.class), mo);
   }

   private void addDsProperties(ManagedObjectImpl mo)
   {
      addCommonProperties(mo);
      addManagedProperty("transaction-isolation", "The transaction isolation level", false, SimpleMetaType.STRING, mo);
      addManagedProperty("user-name", "The username for the connection-url", false, SimpleMetaType.STRING, mo);
      addManagedProperty("password", "The password for the connection-url", false, SimpleMetaType.STRING, mo);
      addManagedProperty("new-connection-sql", "", false, SimpleMetaType.STRING, mo);
      addManagedProperty("check-valid-connection-sql", "", false, SimpleMetaType.STRING, mo);
      addManagedProperty("valid-connection-checker-class-name", "", false, SimpleMetaType.STRING, mo);
      addManagedProperty("exception-sorter-class-name", "", false, SimpleMetaType.STRING, mo);
      addManagedProperty("stale-connection-checker-class-name", "", false, SimpleMetaType.STRING, mo);
      addManagedProperty("track-statements", "", false, SimpleMetaType.STRING, mo);
      addManagedProperty("prepared-statement-cache-size", "", false, SimpleMetaType.INTEGER, new Integer(0), mo);
      addManagedProperty("share-prepared-statements", "", false, SimpleMetaType.BOOLEAN, Boolean.FALSE, mo);
      addManagedProperty("set-tx-query-timeout", "", false, SimpleMetaType.BOOLEAN, Boolean.FALSE, mo);
      addManagedProperty("query-timeout", "", false, SimpleMetaType.INTEGER, new Integer(0), mo);
      addManagedProperty("url-delimiter", "", false, SimpleMetaType.STRING, mo);
      addManagedProperty("url-selector-strategy-class-name", "", false, SimpleMetaType.STRING, mo);
   }

   private void addCommonProperties(ManagedObjectImpl mo)
   {
      // perhaps these descriptions should be externalized in some way
      addManagedProperty("jndi-name", "The jndi name to bind the DataSource under", true, true, SimpleMetaType.STRING, "", mo);
      addManagedProperty("rar-name", "The resource adapter archive name", true, SimpleMetaType.STRING, mo);
      addManagedProperty("use-java-context", "Whether to bind the connection factory under 'java:' context", true, SimpleMetaType.BOOLEAN, Boolean.TRUE, mo);
      addManagedProperty("connection-definition", "The connection factory class name", true, SimpleMetaType.STRING, mo);
      addManagedProperty("jmx-invoker-name", "The name of the JMX invoker", true, SimpleMetaType.STRING, mo);
      addManagedProperty("min-pool-size", "The min size of the pool", true, SimpleMetaType.INTEGER, new Integer(0), mo);
      addManagedProperty("max-pool-size", "The max size of the pool", true, SimpleMetaType.INTEGER, new Integer(10), mo);
      addManagedProperty("blocking-timeout-millis", "The time to wait for a connection to become available before giving up", true, SimpleMetaType.LONG, new Long(30000), mo);
      addManagedProperty("idle-timeout-minutes", "The idle timeout in minutes", true, SimpleMetaType.INTEGER, new Integer(30), mo);
      addManagedProperty("prefill", "Whether to prefill the pool", true, SimpleMetaType.BOOLEAN, mo);
      addManagedProperty("background-validation", "Whether to use backgroup validation", true, SimpleMetaType.BOOLEAN, Boolean.FALSE, mo);
      addManagedProperty("background-validation-minutes", "", false, SimpleMetaType.INTEGER, new Integer(0), mo);
      addManagedProperty("validate-on-match", "", true, SimpleMetaType.BOOLEAN, Boolean.TRUE, mo);
      addManagedProperty("use-strict-min", "", true, SimpleMetaType.BOOLEAN, mo);
      addManagedProperty("no-tx-separate-pools", "", true, SimpleMetaType.BOOLEAN, mo);
      addManagedProperty("statistics-formatter", "", true, SimpleMetaType.STRING, mo);
      addManagedProperty("isSameRM-override-value", "", true, SimpleMetaType.BOOLEAN, mo);
      addManagedProperty("track-connection-by-tx", "", true, SimpleMetaType.BOOLEAN, mo);
      addManagedProperty("config-property", "The connection factory config properties", false, METATYPE_FACTORY.resolve(getMapOfMapsType()), mo);
      addManagedProperty("security-domain", "The security-domain used to validate connections", false, SimpleMetaType.STRING, mo);
      addManagedProperty("depends", "", false, METATYPE_FACTORY.resolve(List.class), mo);
      addManagedProperty("metadata", "", false, METATYPE_FACTORY.resolve(DBMSMetaData.class), mo);
      addManagedProperty("type-mapping", "", true, SimpleMetaType.STRING, mo);
      addManagedProperty("local-transaction", "", true, SimpleMetaType.BOOLEAN, mo);
   }

   private void addManagedProperty(String fieldName,
                                   String fieldDescr,
                                   boolean mandatory,
                                   MetaType metaType,
                                   ManagedObjectImpl mo)
   {
      addManagedProperty(fieldName, fieldDescr, mandatory, false, metaType, null, mo);
   }

   private void addManagedProperty(String fieldName,
         String fieldDescr,
         boolean mandatory,
         MetaType metaType,
         java.io.Serializable value,
         ManagedObjectImpl mo)
   {
      addManagedProperty(fieldName, fieldDescr, mandatory, false, metaType, value, mo);
   }
   private void addManagedProperty(String fieldName,
                                   String fieldDescr,
                                   boolean mandatory,
                                   boolean isID,
                                   MetaType metaType,
                                   java.io.Serializable value,
                                   ManagedObjectImpl mo)
   {
      DefaultFieldsImpl fields = new DefaultFieldsImpl();
      setFieldName(fieldName, fields);
      fields.setDescription(fieldDescr);
      fields.setMandatory(mandatory);
      fields.setMetaType(metaType);
      ManagedPropertyImpl mp = new ManagedPropertyImpl(mo, fields);
      if(isID)
      {
         Map<String, Annotation> annotations = new HashMap<String, Annotation>();
         Map<String, Object> idFields = Collections.emptyMap();
         try
         {
            ManagementObjectID id = (ManagementObjectID) AnnotationProxy.createProxy(idFields, ManagementObjectID.class);
            annotations.put(ManagementObjectID.class.getName(), id);
            mp.setAnnotations(annotations);
         }
         catch(Exception e)
         {
            throw new UndeclaredThrowableException(e);
         }
      }
      super.addProperty(mp);
      if(value != null)
         mp.setValue(value);
   }

   protected void setFieldName(String name, Fields f)
   {
      f.setField(Fields.NAME, name);
      if( propertyNameMappings != null )
      {
         String mappedName = propertyNameMappings.get(name);
         if( mappedName != null )
            f.setField(Fields.MAPPED_NAME, mappedName);
      }
   }

   private Type getMapOfMapsType()
   {
      if(mapOfMapsType == null)
      {
         try
         {
            mapOfMapsType = getClass().getMethod("mapOfMaps").getGenericReturnType();
         }
         catch(NoSuchMethodException e)
         {
            throw new IllegalStateException("Failed to find compoditeValueMap method.");
         }
      }
      return mapOfMapsType;
   }

   private Type getMapType()
   {
      if(mapType == null)
      {
         try
         {
            mapType = getClass().getMethod("compositeValueMap").getGenericReturnType();
         }
         catch(NoSuchMethodException e)
         {
            throw new IllegalStateException("Failed to find compoditeValueMap method.");
         }
      }
      return mapType;
   }

   public Map<String, String> compositeValueMap()
   {
      return null;
   }

   public Map<String, Map<String, String>> mapOfMaps()
   {
      return null;
   }
}
