/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.mx.loading;

import EDU.oswego.cs.dl.util.concurrent.ConcurrentReaderHashMap;

import org.jboss.classloading.spi.ClassLoadingDomain;
import org.jboss.classloading.spi.DomainClassLoader;
import org.jboss.classloading.spi.Translator;
import org.jboss.logging.Logger;
import org.jboss.mx.server.ServerConstants;

import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Vector;

/**
 * Port of legacy LoaderRepository to a ClassLoadingDomain base.
 * 
 * @author  <a href="mailto:juha@jboss.org">Juha Lindfors</a>.
 * @author Scott.Stark@jboss.org
 * @version $Revision: 57108 $
 */
public abstract class LoaderRepositoryDomain
   implements ServerConstants, ClassLoadingDomain
{

   // Attributes ----------------------------------------------------
   protected static Vector loaders = new Vector();
   protected static LoaderRepositoryDomain instance = null;
   protected Translator translator = null;

   /** The loaded classes cache, HashMap<String, Class>.
    * Access synchronized via this.classes monitor.
    */
   private ConcurrentReaderHashMap classes = new ConcurrentReaderHashMap();


   /**
    * Native signature to class map
    */
   private static HashMap nativeClassBySignature;

   // Static --------------------------------------------------------
   private static final Logger log = Logger.getLogger(LoaderRepositoryDomain.class);

   /**
    * Construct the native class map
    */
   static
   {
      nativeClassBySignature = new HashMap();
      nativeClassBySignature.put("boolean", boolean.class);
      nativeClassBySignature.put("byte", byte.class);
      nativeClassBySignature.put("char", char.class);
      nativeClassBySignature.put("double", double.class);
      nativeClassBySignature.put("float", float.class);
      nativeClassBySignature.put("int", int.class);
      nativeClassBySignature.put("long", long.class);
      nativeClassBySignature.put("short", short.class);
      nativeClassBySignature.put("void", void.class);

      nativeClassBySignature.put("boolean[]", boolean[].class);
      nativeClassBySignature.put("byte[]", byte[].class);
      nativeClassBySignature.put("char[]", char[].class);
      nativeClassBySignature.put("double[]", double[].class);
      nativeClassBySignature.put("float[]", float[].class);
      nativeClassBySignature.put("int[]", int[].class);
      nativeClassBySignature.put("long[]", long[].class);
      nativeClassBySignature.put("short[]", short[].class);
   }

   // Public --------------------------------------------------------
   public Vector getLoaders()
   {
      return loaders;
   }

   public URL[] getURLs()
   {
      return null;
   }

   public Class getCachedClass(String classname)
   {
       return (Class)classes.get(classname);
   }

   public Translator getTranslator()
   {
      return translator;
   }

   public void setTranslator(Translator t)
   {
      translator = t;
   }

   /**
    * Compare two loader repository, by default we do no special ordering
    * 
    * @param lr the loader repository
    * @return -1, 0, 1 depending upon the order
    */
   public int compare(LoaderRepositoryDomain lr)
   {
      if (lr == this)
         return 0;
      else
         return 1;
   }
   
   // BEGIN ClassLoadingDomain **************************************************************************************

   /**
    * Loads a class from the repository. This method attempts to load the class
    * using all the classloader registered to the repository.
    *
    * @param className the class to load
    * @return the found class
    * @exception ClassNotFoundException when there is no such class
    */
   public abstract Class loadClass(String className) throws ClassNotFoundException;
   /**
    * Load a class from this domain
    * 
    * @param name the class to load
    * @param resolve whether to resolve the class
    * @param classLoader the requesting classloader
    * @return the class
    * @throws ClassNotFoundException when the class is not found
    */
   public abstract Class loadClass(String name, boolean resolve, DomainClassLoader classLoader) throws ClassNotFoundException;
   
   // END ClassLoadingDomain ****************************************************************************************

   /** Create DomainClassLoader and optionally add it to the repository
    * @param url the URL to use for class loading
    * @param addToRepository a flag indicating if the CL should be added to
    *    the repository
    * @return the UCL instance
    * @throws Exception
    */
   public abstract DomainClassLoader newClassLoader(final URL url, boolean addToRepository)
      throws Exception;
   /** Create DomainClassLoader and optionally add it to the repository
    * @param url the URL to use for class loading
    * @param origURL an orignal URL to use as the URL for the CL CodeSource.
    * This is useful when the url is a local copy that is difficult to use for
    * security policy writing.
    * @param addToRepository a flag indicating if the CL should be added to
    *    the repository
    * @return the CL instance
    * @throws Exception
    */
   public abstract DomainClassLoader newClassLoader(final URL url, final URL origURL,
      boolean addToRepository)
      throws Exception;

   /** Find a resource URL for the given name
    *
    * @param name the resource name
    * @param cl the requesting class loader
    * @return The resource URL if found, null otherwise
    */
   public abstract URL getResource(String name, DomainClassLoader cl);
   /** Find all resource URLs for the given name. Since this typically
    * entails an exhuastive search of the repository it can be a relatively
    * slow operation.
    *
    * @param name the resource name
    * @param cl the requesting class loader
    * @param urls a list into which the located resource URLs will be placed
    */
   public abstract void getResources(String name, DomainClassLoader cl, List urls);

   /** Add a class loader to the repository
    */
   public abstract void addClassLoader(DomainClassLoader cl);
   /** Update the set of URLs known to be associated with a previously added
    * class loader.
    *
    * @param cl
    * @param url
    */
   public abstract boolean addClassLoaderURL(DomainClassLoader cl, URL url);
   /** Remove a cladd loader from the repository.
    * @param cl
    */
   public abstract void removeClassLoader(DomainClassLoader cl);

   /**
    * Return the class of a java native type
    * @return the class, or null if className is not a native class name
    */
   public static final Class getNativeClassForName(String className)
   {
      // Check for native classes
      return (Class)nativeClassBySignature.get(className);
   }

   /**
    * Allow subclasses to override the ordering
    * 
    * @param lr the loader repository
    * @return -1, 0, 1 depending upon the order
    */
   protected int reverseCompare(LoaderRepositoryDomain lr)
   {
      return 0;
   }

   /**
    * Add a class to the the cache
    */
   void cacheLoadedClass(String name, Class cls, DomainClassLoader cl)
   {
       synchronized( classes )
       {
          // Update the global cache
          classes.put(name, cls);
          if( log.isTraceEnabled() )
          {
             log.trace("cacheLoadedClass, classname: "+name+", class: "+cls
                +", cl: "+cl);
          }

           /**
            * TODO: Adding this implementation is a hack for jmx 1.2 checkin.
            * Had to add this because need getCachedClass() to work.
            * However, this method does not add loaded classes to collection
            * to be unloaded when called to remove classloader.
            * Hopefully this will be a short term workaround.
            * Contact telrod@e2technologies.net if you have questions. -TME
            */
       }
   }

   void clear()
   {
      classes.clear();
   }
}
